# run_skeleton_task.py (Filled for Exploration of Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only externally defined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # ========== Exploration Phase (Predicate Discovery) ==========

        # FEEDBACK: (robot-at dice1_pose)
        # The predicate (robot-at dice1_pose) was observed, but may not match the robot-at's type/arity in the main domain.
        # Exploration domain uses: (robot-at ?r - robot ?loc - location)
        # Main domain uses:       (robot-at ?p - location)   (NO robot argument!)
        # The missing predicate may be that robot-at is defined wrongly in the main domain (should be robot-at <robot> <location>)

        # For demonstration/exploration purposes, let's check if the robot being 'at' a location is recognized by the main domain.
        # Try a minimal exploration: use only available skills to bring the robot to 'dice1_pose' (or an available location from positions) and observe behavior.

        # --- Step 1: Choose an available location to move to. ---
        # We'll search for a key that resembles 'dice1_pose', or otherwise any available location.
        target_location = None
        for key in positions:
            if "dice1_pose" in key:
                target_location = key
                break
        if not target_location:
            # Fallback: pick first location in positions
            target_location = next(iter(positions))

        print(f"[Exploration] Attempting to move robot to location: {target_location}")

        # --- Step 2: Determine robot's current location. ---
        # Assume robot-at always at exactly one location. Try to parse from current observation or positions.
        # For demonstration, assume initial location is in positions as well:
        robot_current_location = None
        for key in positions:
            if "robot" in key or "base" in key:
                robot_current_location = key
                break
        if not robot_current_location:
            # If we can't find a robot location, use the target_location (robot is already there)
            robot_current_location = target_location

        # --- Step 3: Move the robot using execute_go skill (if available). ---
        try:
            print(f"[Exploration] Robot moving from {robot_current_location} to {target_location} using execute_go skill...")
            obs, reward, done, info = execute_go(
                env,
                task,
                from_location=robot_current_location,
                to_location=target_location
            )
        except Exception as e:
            # Skill signature mismatch (such as parameter names), try positional form
            try:
                obs, reward, done, info = execute_go(env, task, robot_current_location, target_location)
            except Exception as e2:
                print(f"[Exploration] execute_go failed: {e2}")
                obs, reward, done, info = None, 0, False, {}

        # --- Step 4: Check for observable change or error message ---
        if obs is not None:
            print(f"[Exploration] Robot should now be at: {target_location}")
        else:
            print("[Exploration] Observation after move is None -- check if location predicate is mismatched.")

        # === Main Task Plan Execution (if any) ===
        # Since the purpose here is to explore and surface issues with the predicates (from feedback + PDDL),
        # we pause here. In a typical run, here is where you'd proceed with other skill invocations
        # e.g., execute_pick, execute_place, etc, following an oracle plan.

        # If you wish to try a pick or place, you would call:
        #   obs, reward, done, info = execute_pick(env, task, object_id, location)   # example only

        # --- End of exploration phase ---
        
        print("===== Exploration phase completed. If errors related to robot-at predicate appear, please update domain definition accordingly! =====")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()