# run_skeleton_task.py

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only the externally provided skill functions.

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with knowledge exploration phase.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Exploration Phase - Identify Missing Predicate
        # Feedback: (robot-at drawer_top_place_left)
        # The system expects the predicate (robot-at <location>) to correctly describe where the robot is.
        # This predicate may be missing or uninitialized. The exploration domain suggests we need to
        # ensure the robot's actual positional predicate is reflected and can be sensed.

        # We'll explore with "execute_go" or similar (as execute_go_identify doesn't exist
        # in primitive_skills_static_lock_v2, but execute_go does), and carefully monitor the robot's arrival.
        # Since the skill_code is provided externally, we simply make the call and handle errors.

        # Try to move the robot to drawer_top_place_left if it is not already there
        target_location = 'drawer_top_place_left'
        robot_current_location = None
        # Attempt to infer robot's current position from available positions, if possible
        # Since positions may not contain the robot directly (unless provided), we may be forced to
        # assume starting location or use the observation/feedback.
        # For the purposes of this skeleton, we'll proceed with a plausible default.
        known_locations = []
        for key in positions:
            if 'place' in key:
                known_locations.append(key)
        # Fallback: if nothing found, just use the first two location names for demo
        # In a real codebase, parsing the observation/init would give us the real locations.
        if len(known_locations) == 0:
            known_locations = ['drawer_top_place_right', 'drawer_top_place_left']
        if target_location not in known_locations:
            known_locations.append(target_location)
        # Assume starting position is the first known location, unless otherwise specified
        if target_location in known_locations and len(known_locations) > 1:
            source_location = known_locations[0]
            if source_location == target_location and len(known_locations) > 2:
                source_location = known_locations[1]
        else:
            source_location = 'unknown_location'

        print(f"[Exploration] Attempting to move robot from {source_location} to {target_location}...")

        # Execute the move using an available skill
        try:
            # execute_go(from, to)
            obs, reward, done = execute_go(
                env,
                task,
                from_location=source_location,
                to_location=target_location,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print(f"[Exploration] Robot should now be at {target_location}.")
        except Exception as e:
            print(f"[Exploration] Could not execute 'execute_go': {e}")

        # At this point, we would check/validate if the predicate (robot-at drawer_top_place_left)
        # is now observed in the system's internal state, which would confirm the missing predicate.

        # === Complete Oracle Plan (if it existed) - Demo placeholder ===
        # Normally, here we would proceed to perform primitive actions from the oracle plan step by step:
        # For example:
        #
        # obs, reward, done = execute_pick(env, task, obj, location, ...)
        # obs, reward, done = execute_place(env, task, obj, drawer, location, ...)
        # etc.
        #
        # Always check for errors and print meaningful feedback at each step.
        # For this task, the focus is on the exploration phase for missing predicate.
        #
        # You can expand the next steps here when the actual plan (sequence of skill invocations) is provided.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
