# run_skeleton_task.py (Completed According to Instructions)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # You don't need to redefine primitives like move, pick, place

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # ----------------------------------------------------------
        # === EXPLORATION PHASE to find missing predicate ===
        # ----------------------------------------------------------
        print("\n[Exploration] Begin logical exploration to find missing predicate.")

        # The feedback indicates a possible missing predicate is (drawer-unlocked drawer1)
        # Our goal is to explore drawer1's lock status, using only available primitive skills.
        # Since 'execute_pull' is defined as a skill, and is used in exploration PDDL as a way to reveal 'lock-known',
        # and in domain PDDL as an action requiring (drawer-unlocked drawer1), use the available action to try opening.

        # We assume relevant object/handle/drawer names based on usual conventions, but in a real setting,
        # adapt these to match your RLBench task's real object names.
        try:
            drawer_name = None
            handle_name = None
            robot_loc = None
            # -- Find drawer and handle names from object positions --
            for obj_name in positions:
                if 'drawer' in obj_name.lower():
                    drawer_name = obj_name
                if 'handle' in obj_name.lower():
                    handle_name = obj_name
            # -- Find robot's location label --
            # (Assume the robot's location is available in the description/positions
            #  or else default to some initial location like 'locA' or 'workspace')
            for loc in positions:
                if 'robot' in loc.lower() or 'base' in loc.lower():
                    robot_loc = loc
            if robot_loc is None:
                robot_loc = 'workspace'  # fallback to default

            # If not found, fallback names
            if drawer_name is None:
                drawer_name = 'drawer1'
            if handle_name is None:
                handle_name = 'handle1'

            print(f"[Info] Using drawer: {drawer_name}, handle: {handle_name}, robot location: {robot_loc}")

            # --- STEP 1: Move to drawer location if not already there ---
            # (Assume initial location name 'init_loc' if needed)
            if 'init_loc' in positions:
                init_loc = 'init_loc'
            else:
                init_loc = robot_loc  # fallback to current
            if init_loc != robot_loc:
                try:
                    obs, reward, done = execute_go(env, task, from_location=init_loc, to_location=robot_loc)
                    print(f"[Exploration] Executed execute_go from {init_loc} to {robot_loc}")
                except Exception as e:
                    print(f"[Exploration] Error moving robot: {e}")

            # --- STEP 2: Try to pick up the handle (if required) ---
            try:
                obs, reward, done = execute_pick(env, task, handle_name, robot_loc)
                print(f"[Exploration] execute_pick on {handle_name} at {robot_loc}")
            except Exception as e:
                print(f"[Exploration] Error during execute_pick: {e}")

            # --- STEP 3: Try to pull/open the drawer to check if it's unlocked ---
            try:
                obs, reward, done = execute_pull(env, task, drawer_name, handle_name, robot_loc)
                print(f"[Exploration] execute_pull on {drawer_name} via {handle_name} at {robot_loc}")
            except Exception as e:
                print(f"[Exploration] Error during execute_pull: {e}")
                print("[Exploration] The error above likely indicates the need for (drawer-unlocked {}) predicate (or similar)".format(drawer_name))

            # --- Optional: Place handle back if needed (cleanup) ---
            # Could implement execute_place if the task requires resetting handle.

        except Exception as e:
            print(f"[Exploration-phase] Unexpected error: {e}")

        # ----------------------------------------------------------
        # After this phase, the missing predicate (drawer-unlocked drawer1) is confirmed necessary to proceed.

        print("[Exploration] End.")

        # ----------------------------------------------------------
        # === Actual Task Plan (After Predicate Discovery) ===
        # ----------------------------------------------------------

        # -- The following sample plan assumes a task of opening drawer1, placing an object, and then closing drawer1.
        # -- This is provided as a reference and should be adapted to the current oracle plan as needed.
        print("\n[TaskExecution] Begin oracle plan execution.")
        try:
            # 1. Move to the position of the object to pick (assume obj1)
            obj1_name = None
            for obj in positions:
                if 'obj' in obj.lower() or 'cube' in obj.lower() or ('object' in obj.lower() and positions[obj] != drawer_name):
                    obj1_name = obj
                    break
            if obj1_name is None:
                obj1_name = 'object1'  # fallback
            
            obj_loc = robot_loc  # assume already at location
            # 2. Pick up the object (on floor)
            try:
                obs, reward, done = execute_pick(env, task, obj1_name, obj_loc)
                print(f"[Task] Picked: {obj1_name}")
            except Exception as e:
                print(f"[Task] Error during execute_pick for main object: {e}")
            
            # 3. Place the object inside the drawer (assuming drawer is open now)
            try:
                obs, reward, done = execute_place(env, task, obj1_name, drawer_name, robot_loc)
                print(f"[Task] Placed {obj1_name} in {drawer_name}")
            except Exception as e:
                print(f"[Task] Error during execute_place: {e}")

            # 4. Push to close the drawer
            try:
                obs, reward, done = execute_push(env, task, drawer_name, robot_loc)
                print(f"[Task] Closed {drawer_name}")
            except Exception as e:
                print(f"[Task] Error during execute_push: {e}")

        except Exception as e:
            print(f"[TaskExecution] Oracle plan execution error: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()