# run_skeleton_task.py (Completed to include exploration logic)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use: execute_pick, execute_place, execute_push, execute_pull, execute_sweep, execute_rotate, execute_go, execute_gripper

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()
        init_video_writers(obs)

        # For video/debug recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Discover missing predicate/functionality ===
        # The feedback indicates task planning failed, perhaps due to an unknown or missing predicate.
        
        # The goal is to probe for missing knowledge using available actions.
        # Since the available skills do NOT include special "exploration" skills (from the exploration knowledge PDDL),
        # we have to use only provided primitive skills and observe responses/state changes.

        # Repeatedly attempt to use each skill to gather their effects and feedback,
        # looking for any failures indicative of missing predicates.

        # Skill names to try
        skill_names = [
            'execute_pick',
            'execute_place',
            'execute_push',
            'execute_pull',
            'execute_sweep',
            'execute_rotate',
            'execute_go',
            'execute_gripper'
        ]

        # Log available objects and their locations for parameter choices
        # The exact key structure in 'positions' depends on the environment;
        # Let's assume positions = {'object_name': (x, y, z), ...}
        object_names = list(positions.keys())
        # Dummy values for drawers and locations for illustration
        drawer_names = [obj for obj in object_names if 'drawer' in obj]
        handle_names = [obj for obj in object_names if 'handle' in obj]
        location_names = list(set([tuple(pos) for pos in positions.values()]))

        # ——— Smart param guess: for the sake of code robustness
        # We'll just iterate some guesses for skills parameters
        # (as we don't have type info for each object, extract from names and try-catch failures)
        # This will also serve to probe which predicate or object is missing by raising/logging errors.

        print("=== [Exploration phase: trying built-in primitive skills to probe environment] ===")
        for skill in skill_names:
            try:
                if skill == 'execute_pick':
                    # Try to pick an object on the floor at a known location
                    for obj in object_names:
                        if 'drawer' in obj or 'handle' in obj:
                            continue  # Skip drawers/handles for pick
                        # Guess a plausible location from positions
                        loc = positions.get(obj, None)
                        if loc is not None:
                            print(f"[Exploring] execute_pick: Trying pick({obj}) at {loc}")
                            obs, reward, done = execute_pick(env, task, obj, loc)
                            if done:
                                print("[Exploration] Task ended! (unexpected early termination)")
                                return
                elif skill == 'execute_place':
                    # Try placing a held object in a drawer at a location
                    for obj in object_names:
                        for drawer in drawer_names:
                            loc = None
                            if obj in positions:
                                loc = positions[obj]
                            elif drawer in positions:
                                loc = positions[drawer]
                            else:
                                # Use any as fallback
                                loc = list(positions.values())[0]
                            print(f"[Exploring] execute_place: Trying place({obj}) into {drawer} at {loc}")
                            try:
                                obs, reward, done = execute_place(env, task, obj, drawer, loc)
                            except Exception as e:
                                # Most likely will fail if preconditions are not met--break inner loop.
                                print(f"[Explore] execute_place failed: {e}")
                                continue
                            if done:
                                print("[Exploration] Task ended! (unexpected early termination)")
                                return
                elif skill == 'execute_push':
                    # Try pushing each drawer at its location
                    for drawer in drawer_names:
                        loc = positions.get(drawer, None)
                        if loc is not None:
                            print(f"[Exploring] execute_push: Trying push({drawer}) at {loc}")
                            try:
                                obs, reward, done = execute_push(env, task, drawer, loc)
                            except Exception as e:
                                print(f"[Explore] execute_push failed: {e}")
                                continue
                            if done:
                                print("[Exploration] Task ended!")
                                return
                elif skill == 'execute_pull':
                    # Try pulling handles (if available) at their location
                    for drawer in drawer_names:
                        for handle in handle_names:
                            loc = positions.get(handle, None) or positions.get(drawer, None)
                            if loc is not None:
                                print(f"[Exploring] execute_pull: Trying pull({drawer}, {handle}) at {loc}")
                                try:
                                    obs, reward, done = execute_pull(env, task, drawer, handle, loc)
                                except Exception as e:
                                    print(f"[Explore] execute_pull failed: {e}")
                                    continue
                                if done:
                                    print("[Exploration] Task ended!")
                                    return
                elif skill == 'execute_go':
                    # Try moving between two locations
                    locs = list(positions.values())
                    if len(locs) >= 2:
                        from_loc = locs[0]
                        to_loc = locs[1]
                        print(f"[Exploring] execute_go: go from {from_loc} to {to_loc}")
                        try:
                            obs, reward, done = execute_go(env, task, from_loc, to_loc)
                        except Exception as e:
                            print(f"[Explore] execute_go failed: {e}")
                            continue
                        if done:
                            print("[Exploration] Task ended!")
                            return
                elif skill == 'execute_sweep':
                    for obj in object_names:
                        if 'drawer' not in obj:
                            loc = positions.get(obj, None)
                            if loc is not None:
                                print(f"[Exploring] execute_sweep: sweep({obj}) at {loc}")
                                try:
                                    obs, reward, done = execute_sweep(env, task, obj, loc)
                                except Exception as e:
                                    print(f"[Explore] execute_sweep failed: {e}")
                                    continue
                                if done:
                                    print("[Exploration] Task ended!")
                                    return
                elif skill == 'execute_rotate':
                    # No parameters? Try to call
                    print("[Exploring] execute_rotate (no params)")
                    try:
                        obs, reward, done = execute_rotate(env, task)
                    except Exception as e:
                        print(f"[Explore] execute_rotate failed: {e}")
                        continue
                    if done:
                        print("[Exploration] Task ended!")
                        return
                elif skill == 'execute_gripper':
                    print("[Exploring] execute_gripper (no params)")
                    try:
                        obs, reward, done = execute_gripper(env, task)
                    except Exception as e:
                        print(f"[Explore] execute_gripper failed: {e}")
                        continue
                    if done:
                        print("[Exploration] Task ended!")
                        return
            except Exception as e:
                print(f"[Exploration-phase] Exception in skill {skill}: {e}")

        print("=== [Exploration phase finished: check execution traces or failure logs for missing predicate hints] ===")
        print("===== End of Skeleton Task =====")

    finally:
        shutdown_environment(env)
