# run_skeleton_task.py (Filled per instructions and feedback)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only pre-defined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = 'object_1': (0,0,0), 'object_2': (1,1,1), ... (dictionary format)
        positions = get_object_positions()
        
        # --- Begin Exploration Phase ---
        # Feedback indicates that (drawer-closed drawer1) is missing: must check/establish if this predicate exists in the world state.
        # Use the available exploration actions/skills.
        # Since only 'execute_pull', 'execute_push', 'execute_pick', etc. are available, we use them to probe the state.
        #
        # We'll assume 'drawer1' is the drawer; params for skills will be guessed from positions if necessary.
        #
        # First, gather all needed names (objects/drawers/handles/locations) from descriptions/object_positions.
        # For generic code, we'll try a few best guesses but must adjust based on what's available.
        #
        # NOTE: Real plan should use an oracle plan, but for this task, include only the exploration step per feedback,
        # making sure to attempt to determine if 'drawer-closed drawer1' is true (or can be made so).
        # Since only predefined skills may be used, attempt to execute e.g. execute_push or execute_pull on drawer1 and handle exceptions as per instructions.

        # Try-catch block for robust exploration
        try:
            # --- Identify drawer and handle object names as needed ---
            # Use dictionary provided by get_object_positions()
            # We'll look for 'drawer1' and its handle. Adjust as necessary.
            drawer_name = None
            handle_name = None
            location_name = None
            # Try matching keys to possible drawer and handle names
            for obj_name in positions.keys():
                if 'drawer1' in obj_name.lower() and ('drawer' in obj_name.lower() or 'draw' in obj_name.lower()):
                    drawer_name = obj_name
                if 'handle' in obj_name.lower() and ('1' in obj_name.lower() or drawer_name and drawer_name[-1] in obj_name):
                    handle_name = obj_name
            # For location, pick any known location, or fallback to None if unknown
            for obj_name, pos in positions.items():
                if 'location' in obj_name.lower():
                    location_name = obj_name
            # Fallbacks for demonstration if names can't be found
            if drawer_name is None:
                drawer_name = 'drawer1'  # Default
            if handle_name is None:
                handle_name = 'handle1'  # Default
            if location_name is None:
                location_name = 'location1'  # Default

            # Print discovered names
            print(f"[Exploration] Using drawer: {drawer_name}, handle: {handle_name}, location: {location_name}")            

            # --- Perform Exploration/Test of Drawer State ---
            # We'll attempt to execute 'execute_push' and handle the feedback to detect if the drawer is closed.
            # If 'execute_push' fails because the drawer is not open (e.g., closed or locked),
            # we infer 'drawer-closed' is true or must be made true.
            #
            # These skills generally take (drawer_name, location_name) as parameters.

            print("[Exploration] Attempting to push (close) the drawer to confirm 'drawer-closed'...")
            try:
                obs, reward, done = execute_push(env, task, drawer_name, location_name)
                print("[Exploration] Successfully pushed (closed) the drawer. Predicate (drawer-closed {}) likely holds.".format(drawer_name))
            except Exception as e:
                print("[Exploration] Could not push (close) the drawer. Likely already closed or not in suitable state. Exception:", e)
                print("[Exploration] Therefore, predicate (drawer-closed {}) likely already holds!".format(drawer_name))
            
            # Alternatively, to test opening: use execute_pull (if holding the handle)
            # This requires first picking the handle (execute_pick), then execute_pull
            print("[Exploration] Attempting to pick the handle, then pull the drawer to test if it can be opened (to probe locked/closed state)...")
            # Pick the handle
            try:
                hand_pos = positions.get(handle_name, None)
                loc_for_handle = location_name
                if hand_pos is None:
                    hand_pos = (0,0,0)
                # Try picking the handle
                obs, reward, done = execute_pick(env, task, handle_name, loc_for_handle)
                print("[Exploration] Successfully picked handle:", handle_name)
                # Now try to pull on the drawer with the handle
                obs, reward, done = execute_pull(env, task, drawer_name, handle_name, location_name)
                print("[Exploration] Successfully pulled the drawer open. Drawer state is now open (drawer-open {} holds).".format(drawer_name))
            except Exception as e:
                print("[Exploration] Could not pick handle or pull the drawer. Exception:", e)
                print("[Exploration] Reinforces that drawer may be closed and locked/closed.")

        except Exception as exp_explore:
            print("[Exploration] Fatal exploration exception:", exp_explore)
        
        # ---- END EXPLORATION ----
        # After exploration, further skills can be used as needed for the main oracle plan
        # but for this code, we focus on the exploration per instructions and feedback.

        # TODO: Continue with task plan, using only provided skills. This is placeholder for further plan steps.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
