# run_skeleton_task.py (Completed)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills, do not redefine

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Identify known objects and locations from the positions dict
        # This mapping will depend on the environment's state. We'll use some example names.
        # Please adapt these with the correct keys for your environment.
        robot_initial_pos = positions.get('robot', None)
        drawer1_pos = positions.get('drawer1', None)
        handle1_pos = positions.get('handle1', None)
        floor_obj_pos = None
        for obj_name, pos in positions.items():
            if "obj" in obj_name or "object" in obj_name:
                floor_obj_pos = pos
                floor_obj_name = obj_name
                break

        # --- Exploration Phase for Missing Predicate ---
        print("[Exploration] Attempting discovery of missing predicate...")

        # According to feedback, the missing predicate involves (drawer-closed drawer1)
        # We'll try to discover (drawer-closed drawer1) using available skills and environment info

        # PDDL exploration domain hints: "lock-known" predicate is achieved with execute_pull, but in our available skills
        # we do NOT have the explicit "explore" actions; only the actual manipulation primitives.
        # So, to infer the state (and the missing predicate), we attempt to use the regular skills and use feedback/logging
        # to check their preconditions/effects.

        # We'll attempt to pull (open) the drawer directly.
        # The effect/feedback of this will tell us if (drawer-closed drawer1) was missing.
        # We have 'execute_pull', which according to the domain requires (drawer-closed ?d) as a precondition.

        exploration_done = False
        
        # Prepare the parameters for execute_pull
        # We assume handle1 is the handle object on drawer1
        # Our plan: try to pick the handle first, then pull (open) the drawer

        try:
            current_pos = positions.get('robot', None)
            handle_obj_name = "handle1"
            drawer_obj_name = "drawer1"
            robot_location = "location1"  # Placeholder; use correct key if locations are named

            # 1. Go to the handle's location if necessary (using 'execute_go')
            # Assuming handle is at the same location as drawer and we are at a different spot
            if robot_initial_pos is not None and handle1_pos is not None:
                if np.linalg.norm(np.array(robot_initial_pos) - np.array(handle1_pos)) > 0.01:
                    print("[Exploration] Moving robot to handle's location for exploration.")
                    obs, reward, done = execute_go(env, task,
                                                   from_location=robot_initial_pos,
                                                   to_location=handle1_pos,
                                                   timeout=10.0)
                    if done:
                        print("[Exploration] Aborted: Task ended during go.")
                        return
            else:
                print("[Exploration] Positions for robot or handle not found, skipping movement.")

            # 2. Pick the handle (execute_pick)
            print("[Exploration] Attempting to pick the handle.")
            obs, reward, done = execute_pick(env, task,
                                             obj=handle_obj_name,
                                             location=handle1_pos,
                                             approach_distance=0.15,
                                             max_steps=100,
                                             threshold=0.01,
                                             approach_axis='z',
                                             timeout=10.0)
            if done:
                print("[Exploration] Aborted: Task ended after pick.")
                return

            # 3. Attempt to pull (i.e., open) the drawer
            print("[Exploration] Attempting to pull (open) the drawer using the handle.")
            obs, reward, done = execute_pull(env, task,
                                             drawer=drawer_obj_name,
                                             handle=handle_obj_name,
                                             location=handle1_pos,
                                             timeout=10.0)
            if done:
                print("[Exploration] Aborted: Task ended after pull.")
                return

            # If all above succeeded, we've discovered the drawer was indeed closed (confirmed predicate)
            exploration_done = True
            print("[Exploration] Success: (drawer-closed drawer1) predicate explored and confirmed.")

        except Exception as e:
            print(f"[Exploration] Exception encountered during exploration: {e}")
            # Could not confirm the predicate in this run, but we handled the error gracefully.

        # Continue with the rest of the main task plan as appropriate
        # ------------------------
        # Generic plan (example, to be replaced with your oracle plan steps):
        # Assume that after exploring, our goal is e.g., place an object in the drawer.
        # Steps: go to object, pick object, go to drawer, open drawer if needed, place object in drawer, close drawer.

        try:
            # 1. Go to the object's location
            if floor_obj_pos is not None and robot_initial_pos is not None:
                print("[Task] Moving to object location:", floor_obj_pos)
                obs, reward, done = execute_go(env, task,
                                               from_location=robot_initial_pos,
                                               to_location=floor_obj_pos,
                                               timeout=10.0)
                if done:
                    print("[Task] Task ended on moving to object.")
                    return

            # 2. Pick the object
            print("[Task] Picking the object:", floor_obj_name)
            obs, reward, done = execute_pick(env, task,
                                             obj=floor_obj_name,
                                             location=floor_obj_pos,
                                             approach_distance=0.15,
                                             max_steps=100,
                                             threshold=0.01,
                                             approach_axis='z',
                                             timeout=10.0)
            if done:
                print("[Task] Task ended after picking object.")
                return

            # 3. Go to drawer location
            if drawer1_pos is not None:
                print("[Task] Moving to drawer location:", drawer1_pos)
                obs, reward, done = execute_go(env, task,
                                               from_location=floor_obj_pos,
                                               to_location=drawer1_pos,
                                               timeout=10.0)
                if done:
                    print("[Task] Task ended on moving to drawer.")
                    return

            # 4. Open the drawer if it's not open (assume previous pull did it, else open here)
            # (already tried in exploration, so likely not needed; skip or check status if available)

            # 5. Place the object in the drawer (execute_place)
            print("[Task] Placing the object into the drawer.")
            obs, reward, done = execute_place(env, task,
                                              obj=floor_obj_name,
                                              drawer=drawer_obj_name,
                                              location=drawer1_pos,
                                              timeout=10.0)
            if done:
                print("[Task] Task ended after placing object.")
                return

            # 6. Push/close the drawer (execute_push)
            print("[Task] Closing the drawer.")
            obs, reward, done = execute_push(env, task,
                                             drawer=drawer_obj_name,
                                             location=drawer1_pos,
                                             timeout=10.0)
            if done:
                print("[Task] Task ended after closing drawer.")
                return

        except Exception as e:
            print(f"[Task] Exception during main plan execution: {e}")

        print("[Task] Plan execution completed.")
        # ------------------------

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()