# run_skeleton_task.py

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Predefined primitive skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Example keys, adjust if necessary
        # For exploration, we need to find if "robot-at" relates to a location not yet known or detected
        # Feedback: (robot-at drawer_top_place_left)
        # This suggests we may need to move the robot to 'drawer_top_place_left' and see if a missing predicate emerges.

        # List all locations from positions dictionary
        location_names = [k for k in positions if 'place' in k or 'drawer' in k or 'table' in k or 'loc' in k]

        # Initial assumption: robot starts at some known initial location
        # For exploration, let's try moving the robot to 'drawer_top_place_left' (if it exists)
        # and check if we can detect some missing predicate, as per feedback

        # Step 1: Find current robot location (try 'robot' or 'robot_base' key, adjust as necessary)
        robot_location = None
        for k in positions:
            if 'robot' in k and ('pos' in k or 'base' in k):
                robot_location = k
                break
        if robot_location is None:
            # Fallback: some environments use known start locations
            robot_location = 'robot_base'
        if robot_location not in positions:
            # Just grab the first location as fallback
            robot_location = location_names[0] if location_names else list(positions.keys())[0]

        # Step 2: Try to move the robot to 'drawer_top_place_left'
        # (Assume 'drawer_top_place_left' is a valid location and exists in positions)
        target_location = 'drawer_top_place_left'
        if target_location not in positions:
            # Try similar name fallback
            for name in location_names:
                if 'drawer_top' in name and 'left' in name:
                    target_location = name
                    break
            else:
                print("[Exploration] Could not find drawer_top_place_left!")
                target_location = location_names[0]

        # Step 3: Call the move (execute_go) skill to go from robot_location to target_location
        print(f"[Exploration] Moving robot from '{robot_location}' to '{target_location}' to test (robot-at ...) predicate.")

        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_location=positions[robot_location],
                to_location=positions[target_location],
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print(f"[Exploration] Robot reached '{target_location}'.")
        except Exception as e:
            print(f"[Error] Could not execute_go from {robot_location} to {target_location}: {e}")
            obs = None

        # Step 4: After reaching drawer_top_place_left, check or "probe" for any missing facts/predicates
        # (In a full implementation, you'd check task state or observation for missing predicate, e.g. 'robot-at drawer_top_place_left')
        # For now, after the move, let's print a message and simulate the exploration phase.
        #
        # If your task/obs provides a 'predicates' or 'facts' member, you could inspect it here and print missing facts.
        #
        # Example:
        #   if hasattr(task, 'state_predicates'):
        #       preds = task.state_predicates()
        #       if 'robot-at drawer_top_place_left' not in preds:
        #           print("[Exploration] Found missing predicate after move!")

        print("[Exploration] Exploration phase complete. Please check your logs/observations to identify missing predicates.")

        # === End of Exploration ===
        # Now, continue to your actual plan execution if any, using only predefined skills
        #
        # Example: (if you needed to pick, place, push, pull, etc. at that location)
        #
        # try:
        #     obs, reward, done = execute_pick(env, task, object_name, target_location_pos, ...)
        # except Exception as e:
        #     print(f"[Error] Could not execute_pick: {e}")
        #

        # TODO: Insert remaining oracle plan steps here using only predefined skill functions,
        # and always handle exceptions gracefully.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
