# run_skeleton_task.py (Completed Based on Exploration and Feedback)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skill functions

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, including exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # This will be used if we need to know where objects are.
        # Example: object_pos = positions['dice2']

        # === Exploration Phase for Missing Predicates ===

        # --- Feedback Provided: (on-floor dice2) ---
        # We are missing a predicate that recognizes 'on-floor' for dice2.
        # Exploration knowledge suggests we need to investigate object properties via available skills.

        # 1. Use execute_sweep (if available) or other skills to identify or discover "on-floor" property.
        # 2. Use the skill functions ("execute_pick", etc.) directly to test which skill is successful.
        # 3. Log the results for discovering the missing predicate.

        # Available skills relevant for this exploration:
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # For this domain, "on-floor" is likely related with being able to pick up the object from the floor using 'execute_pick'.
        # We'll attempt to pick dice2 from every location until successful.

        # ---- Exploration Start ----
        # List all objects and locations. Since objects/locations are not specified in observation, let's assume 'dice2' is one relevant object.

        object_names = list(positions.keys())
        dice2_name = None
        for name in object_names:
            if "dice2" in name:
                dice2_name = name
                break
        # If 'dice2' can't be found, just use the first object
        if dice2_name is None and object_names:
            dice2_name = object_names[0]

        # Assume locations and drawers must be known or given in env/task/object_positions
        # Let's extract location and drawer candidates
        locations = []
        drawers = []
        for name, pos in positions.items():
            if "room" in name or "loc" in name or "floor" in name:
                locations.append(name)
            if "drawer" in name:
                drawers.append(name)
        # Fallbacks:
        if not locations:
            # Fallback to any objects with no drawer/handle in name
            locations = [n for n in object_names if "drawer" not in n and "handle" not in n]

        # Exploration variables for this run
        exploration_results = {}

        # Attempt execute_pick on dice2 at every location
        robot_location = None
        # Find current robot location if possible
        for name, pos in positions.items():
            if "robot" in name or "robby" in name:
                robot_location = name
                break
        if not robot_location and locations:
            robot_location = locations[0]

        if dice2_name and robot_location:
            # Try to execute_pick dice2 at the current robot location
            print(f"[Exploration] Attempting to pick '{dice2_name}' at location '{robot_location}'...")
            try:
                result_pick = execute_pick(env, task, dice2_name, robot_location)
                exploration_results['execute_pick'] = 'success'
                print("[Exploration] execute_pick succeeded (Predicate present: 'on-floor' for {})".format(dice2_name))
            except Exception as e:
                exploration_results['execute_pick'] = f'fail: {e}'
                print("[Exploration] execute_pick failed:", e)
        else:
            print("[Exploration] dice2 or robot_location not found; skipping execute_pick")
            exploration_results['execute_pick'] = 'skipped'

        # Also optionally execute_sweep as a fallback exploration
        if dice2_name and robot_location:
            print(f"[Exploration] Attempting to sweep '{dice2_name}' at '{robot_location}'...")
            try:
                result_sweep = execute_sweep(env, task, dice2_name, robot_location)
                exploration_results['execute_sweep'] = 'success'
                print("[Exploration] execute_sweep succeeded for on-floor investigation.")
            except Exception as e:
                exploration_results['execute_sweep'] = f'fail: {e}'
                print("[Exploration] execute_sweep failed:", e)
        else:
            exploration_results['execute_sweep'] = 'skipped'

        # Optionally: Try to go to another location if available, then retry pick
        for loc in locations:
            if loc == robot_location:
                continue
            print(f"[Exploration] Attempting to move robot to '{loc}'...")
            try:
                result_go = execute_go(env, task, robot_location, loc)
                print(f"[Exploration] execute_go to {loc} succeeded.")
                robot_location = loc
                # Retry pick after moving
                print(f"[Exploration] Attempting to pick '{dice2_name}' at location '{robot_location}'...")
                try:
                    result_pick2 = execute_pick(env, task, dice2_name, robot_location)
                    exploration_results[f'execute_pick@{loc}'] = 'success'
                    print("[Exploration] execute_pick succeeded after move.")
                except Exception as e:
                    exploration_results[f'execute_pick@{loc}'] = f'fail: {e}'
                    print("[Exploration] execute_pick failed after move:", e)
                break  # Exploration complete after one attempt at new location
            except Exception as e:
                print(f"[Exploration] execute_go to {loc} failed:", e)

        print("[Exploration] Results:", exploration_results)

        # ---- Exploration End ----

        # === End of task ===
        # (You may add goal-oriented plans below if available. This run focuses on exploration for missing predicates.)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()