# run_skeleton_task.py (Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use all predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")

    env, task = setup_environment()
    try:
        # Reset and record initial state
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        positions = get_object_positions()

        # === Exploration Phase: Detect missing predicate about drawer-open ===
        # Context: Feedback suggests (drawer-open drawer3) is relevant; we should explore to identify/confirm it.

        # Example: suppose our drawers are drawer1, drawer2, drawer3, and handles
        drawer_name = 'drawer3'
        handle_name = 'handle3'
        robot_location = None
        drawer_location = None
        handle_location = None

        # Try to dynamically resolve locations
        try:
            # Search for location that contains the drawer and its handle
            for loc, objs in positions.get('locations', {}).items():
                if drawer_name in objs:
                    drawer_location = loc
                if handle_name in objs:
                    handle_location = loc
            # For the robot, check where it is
            robot_location = [loc for loc, objs in positions.get('locations', {}).items()
                              if 'robot' in objs or 'robby' in objs]
            if robot_location:
                robot_location = robot_location[0]
            elif positions.get('robot_location') is not None:
                robot_location = positions['robot_location']
            else:
                robot_location = drawer_location  # fallback
        except Exception as e:
            drawer_location = 'drawer3_area'
            handle_location = 'drawer3_area'
            robot_location = 'base'

        print(f"[Exploration] Robot at: {robot_location}, {drawer_name} at: {drawer_location}, {handle_name} at: {handle_location}")

        # We want to verify the open state predicate; exploration is required.
        # We'll check if pulling the handle will provide feedback/evidence of 'drawer-open'.

        # --- 1. Move to drawer location (execute_go) ---
        try:
            if robot_location != drawer_location:
                print(f"[Exploration] Moving robot from {robot_location} to {drawer_location}")
                obs, reward, done = execute_go(env, task, from_location=robot_location, to_location=drawer_location)
                robot_location = drawer_location
        except Exception as e:
            print(f"[Exploration] Error moving robot: {e}")

        # --- 2. Pick the handle to open the drawer (execute_pick) ---
        try:
            print(f"[Exploration] Picking handle: {handle_name}")
            obs, reward, done = execute_pick(env, task, handle_name, drawer_location)
        except Exception as e:
            print(f"[Exploration] Error picking handle: {e}")

        # --- 3. Pull to open the drawer (execute_pull) ---
        try:
            print(f"[Exploration] Pulling to open {drawer_name} with handle {handle_name}")
            obs, reward, done = execute_pull(env, task, drawer_name, handle_name, drawer_location)
            # If success, we deduce predicate (drawer-open drawer3) can be achieved
            print(f"[Exploration] Success: Drawer '{drawer_name}' should now be open.")
        except Exception as e:
            print(f"[Exploration] Error pulling drawer: {e}")

        # --- 4. Restore predicate knowledge if needed ---
        # (At this point, from feedback, the relevant predicate is (drawer-open drawer3))
        print("[Exploration] If planning fails due to missing predicate, likely 'drawer-open' is missing.")

        # --- 5. (Optional) Place an object into the open drawer to trigger higher-level effects ---
        # This step is to show exploration is complete and now main plan can proceed if required.

        # NOTE: In a real plan, use explore results to add predicate to state for downstream planning.

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
