# run_skeleton_task.py

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only provided skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example: positions = {'dice1': (x1, y1, z1), 'dice2': (x2, y2, z2), ...}
        try:
            positions = get_object_positions()
        except Exception as e:
            print("[Error] Could not retrieve object positions:", e)
            positions = {}

        # === Exploration Phase: Identify Missing Predicate ===
        #
        # Feedback indicates "(on-floor dice2)" is relevant, suggesting
        # we need to explore whether specific objects (e.g., dice2) are on the floor.
        #
        # Since there is no explicit exploration skill, we will utilize the available
        # skills to either attempt pick/place or use the environment/task API to
        # check for this predicate.

        # Phase 1: Try to pick up dice2 using execute_pick to test if it has the (on-floor) property.
        # If execute_pick fails, we assume dice2 is not on the floor.

        # Obtain object names relevant to the task
        dice2_name = None
        for k in positions.keys():
            if 'dice2' in k:
                dice2_name = k
                break

        # Determine robot starting location (if needed for skills like execute_go)
        robot_location = None
        if hasattr(task, "get_robot_location"):
            robot_location = task.get_robot_location()
        else:
            # Fallback: try to find a plausible location from observation or positions
            robot_location = None
            for k in positions:
                if "robot" in k:
                    robot_location = k
                    break

        # Assume locations are named 'floor', 'drawer', etc.
        # Try to infer where dice2 is
        dice2_floor_location = None
        if dice2_name and isinstance(positions.get(dice2_name), tuple):
            # We might infer the location by proximity; as a fallback, use a likely name
            dice2_floor_location = "floor"
            # Or, set to a position understood by skill functions

        # Safeguard: use a generic variable if concrete names not available
        # Main Phase: Try picking up dice2 to infer its "on-floor" predicate state

        action_result = None
        missing_predicate_found = False
        print("[Exploration] Attempting to pick up dice2 to check (on-floor dice2) predicate...")
        try:
            if dice2_name and dice2_floor_location:
                # Try using the execute_pick skill
                # Many pick implementations expect: (env, task, target_obj_name/pos, ...)
                # The required signature should match skill_code implementation;
                # For illustrative purpose, use a general call form.

                # Some environments may require additional parameters (e.g., approach_distance, max_steps, etc.)
                obs, reward, done = execute_pick(
                    env,
                    task,
                    target_obj_name=dice2_name,
                    target_pos=positions[dice2_name],
                    approach_distance=0.10,
                    max_steps=50,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                action_result = done
                print("[Exploration] execute_pick for dice2 result:", done)
                if done:
                    print("[Exploration] Predicate (on-floor dice2) verified present!")
                    missing_predicate_found = True
                else:
                    print("[Exploration] Could NOT confirm (on-floor dice2).")
            else:
                print("[Exploration] Could not locate dice2 for exploration.")
        except Exception as e:
            print("[Exploration] Exception during execute_pick on dice2:", e)
            print("[Exploration] Could NOT confirm (on-floor dice2).")

        # Exploration complete - summarize finding
        if missing_predicate_found:
            print("[Exploration] Missing predicate found: (on-floor dice2)")
        else:
            print("[Exploration] No evidence for missing predicate found using execute_pick.")

        # Optionally, now continue with planned task actions
        # For illustration, if exploration succeeded, proceed to place dice2 in a drawer (if drawer exists)
        if missing_predicate_found:
            # Try to find a drawer object
            drawer_name = None
            for k in positions.keys():
                if 'drawer' in k:
                    drawer_name = k
                    break
            if drawer_name:
                # Try to open the drawer (if required), then place the object
                # These skills are provided: execute_pull (to open), execute_place (to place in drawer)
                handle_name = None
                # Search for a handle near drawer (if relevant)
                for k in positions.keys():
                    if 'handle' in k:
                        handle_name = k
                        break
                # First: Approach drawer and open it if not open
                try:
                    # Assume the robot must be at the drawer's location
                    drawer_location = "drawer"
                    if handle_name:
                        obs, reward, done = execute_pick(
                            env,
                            task,
                            target_obj_name=handle_name,
                            target_pos=positions[handle_name],
                            approach_distance=0.10,
                            max_steps=30,
                            threshold=0.01,
                            approach_axis='z',
                            timeout=10.0
                        )
                        if done:
                            print("[Task] Picked up handle:", handle_name)
                            obs, reward, done = execute_pull(
                                env,
                                task,
                                target_drawer_name=drawer_name,
                                target_handle_name=handle_name,
                                target_pos=positions[drawer_name],
                                approach_distance=0.10,
                                max_steps=50,
                                threshold=0.01,
                                approach_axis='z',
                                timeout=10.0
                            )
                            if done:
                                print("[Task] Drawer opened. Attempting to place dice2 into drawer...")
                                # Now place dice2 into the drawer
                                obs, reward, done = execute_place(
                                    env,
                                    task,
                                    target_obj_name=dice2_name,
                                    target_drawer_name=drawer_name,
                                    target_pos=positions[drawer_name],
                                    approach_distance=0.10,
                                    max_steps=50,
                                    threshold=0.01,
                                    approach_axis='z',
                                    timeout=10.0
                                )
                                if done:
                                    print("[Task] Successfully placed dice2 into drawer!")
                                else:
                                    print("[Task] Failed to place dice2 into drawer.")
                            else:
                                print("[Task] Failed to open the drawer.")
                        else:
                            print("[Task] Failed to pick up handle for drawer opening.")
                except Exception as e:
                    print("[Task] Error during drawer open/place sequence:", e)
            else:
                print("[Task] No drawer found in the environment for placement.")

        # Otherwise, insert more steps here as required by further task plans or oracle plans

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()