# run_skeleton_task.py (Enhanced with Exploration Phase for Missing Predicate Detection)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use all predefined skills only

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Enhanced generic skeleton for running any task in your simulation,
       including predicate exploration for missing predicates via available skills.
    '''
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # Reset the task to its initial state and get initial observation
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # ==== Get object/environment layout ====
        positions = get_object_positions()

        # ----- Phase 1: Predicate Exploration -----
        #
        # The plan failed due to a timeout, likely indicating that a precondition is not met,
        # possibly because a predicate (such as lock-known, identified, weight-known, etc.)
        # is missing or not set in the initial state.
        #
        # To explore which predicate is missing, we try available skill actions from the
        # exploration domain and check environment feedback/errors.
        #
        # Note: Only use provided skills (e.g., execute_pick, execute_place, execute_pull, execute_go, etc.)

        print("[Exploration] Starting exploration phase to identify missing predicates...")

        # Helper: For logging failures to help identify missing predicates
        exploration_results = []

        # Get object and location names from positions (basic sample)
        object_names = [name for name, pos in positions.items() if 'drawer' not in name]
        drawer_names = [name for name in positions.keys() if 'drawer' in name]
        location_names = list(positions.keys())  # Fallback: treat each object as a location anchor

        # Find the robot's starting location: hacky!
        robot_loc_name = None
        for name, pos in positions.items():
            if "robot" in name or "base" in name:  # Convention, change as per your system
                robot_loc_name = name
                break
        if robot_loc_name is None:
            robot_loc_name = location_names[0]  # Just pick the first for simplicity

        # Try known skills that give knowledge predicates according to exploration domain:
        # e.g., execute_pull → lock-known, execute_pick_weight → weight-known, execute_pick_durability → durability-known, etc.
        # Assume we have execute_pull, execute_pick, execute_place, execute_go, execute_sweep, execute_push, execute_gripper

        # The "execute_pull" in exploration grants lock-known, so we attempt a pull on a handle or affordance
        
        # The domain also uses execute_pick and execute_go. We'll iterate over accessible objects.

        for obj_name in object_names:
            try:
                print(f"[Exploration] Attempt to execute_pick on: {obj_name} at {robot_loc_name}")
                # Try picking up each object (could fail if not hand-empty, or predicate missing)
                obs, reward, done = execute_pick(env, task, target_object=obj_name, target_pos=positions[obj_name], approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0)
                if done:
                    print(f"[Exploration] Successfully picked up {obj_name} (no missing predicate for pick).")
                    exploration_results.append(("execute_pick", obj_name, True))
                else:
                    print(f"[Exploration] Could not pick up {obj_name}, may indicate missing predicate.")
                    exploration_results.append(("execute_pick", obj_name, False))
            except Exception as e:
                print(f"[Exploration] Exception during execute_pick for {obj_name}: {e}")
                exploration_results.append(("execute_pick", obj_name, False))

        # Try execute_pull, might be necessary to acquire lock-known predicate for drawers/handles
        for handle_obj in object_names:
            for drawer in drawer_names:
                try:
                    print(f"[Exploration] Attempt to execute_pull on drawer {drawer} with handle {handle_obj} at {robot_loc_name}")
                    # This abstraction may vary: check if handle_obj is handle for this drawer (if you have 'handle-of' info)
                    # We'll make a broad attempt
                    obs, reward, done = execute_pull(env, task, drawer_name=drawer, handle_object=handle_obj, target_pos=positions[drawer], approach_distance=0.15, max_steps=100, threshold=0.01, timeout=10.0)
                    if done:
                        print(f"[Exploration] Successfully pulled {handle_obj} of {drawer}.")
                        exploration_results.append(("execute_pull", drawer, handle_obj, True))
                    else:
                        print(f"[Exploration] Could not pull {handle_obj} of {drawer}. Possible missing predicate!")
                        exploration_results.append(("execute_pull", drawer, handle_obj, False))
                except Exception as e:
                    print(f"[Exploration] Exception during execute_pull for {handle_obj} on {drawer}: {e}")
                    exploration_results.append(("execute_pull", drawer, handle_obj, False))

        # Try open/close (push) drawers, if they exist, possibly blocked by missing predicate (e.g. drawer-locked)
        for drawer in drawer_names:
            try:
                print(f"[Exploration] Attempt to execute_push (close) on drawer {drawer} at {robot_loc_name}")
                obs, reward, done = execute_push(env, task, drawer_name=drawer, target_pos=positions[drawer], approach_distance=0.15, max_steps=100, threshold=0.01, timeout=10.0)
                if done:
                    print(f"[Exploration] Successfully pushed (closed) drawer {drawer}.")
                    exploration_results.append(("execute_push", drawer, True))
                else:
                    print(f"[Exploration] Could not push (close) drawer {drawer}.")
                    exploration_results.append(("execute_push", drawer, False))
            except Exception as e:
                print(f"[Exploration] Exception during execute_push for {drawer}: {e}")
                exploration_results.append(("execute_push", drawer, False))

        # Open/close actions or place actions can be further attempted similarly if part of the main plan

        print("[Exploration] Exploration phase complete. Results summarised above.")
        print("[Exploration] If any actions systematically fail due to unmet preconditions, log will indicate which predicate may be missing.")

        # ----- Phase 2: Main Task Execution (Example plan; update to match your scenario) -----
        #
        # After exploration and possible environment state adjustment, attempt to execute the main plan.
        # Replace this with your goal-directed sequence, e.g.:
        #
        # - move robot to target location (execute_go)
        # - pick up object (execute_pick)
        # - open drawer (execute_pull)
        # - place object (execute_place)
        # - close drawer (execute_push)
        #
        # This MUST use only the predefined skills as per the available skill names.

        print("[Task] Starting main task execution sequence...")

        # --- Example main sequence using available skill APIs ---

        try:
            # Step 1: Move to first location with an object (if needed)
            if len(location_names) >= 2:
                start_loc = location_names[0]
                next_loc = location_names[1]
                print(f"[Task] Moving from {start_loc} to {next_loc}")
                obs, reward, done = execute_go(env, task, from_location=start_loc, to_location=next_loc, approach_distance=0.15, max_steps=100, threshold=0.01, timeout=10.0)
                if not done:
                    print(f"[Task] Failed to move from {start_loc} to {next_loc}, check for missing predicates!")
            # Step 2: Try to pick, pull, place, push as needed
            for obj_name in object_names:
                print(f"[Task] Attempting to pick up object: {obj_name}")
                try:
                    obs, reward, done = execute_pick(env, task, target_object=obj_name, target_pos=positions[obj_name], approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0)
                    if done:
                        print(f"[Task] Picked up {obj_name}")
                except Exception as e:
                    print(f"[Task] Exception in main pick for {obj_name}: {e}")

            # (Add custom plan steps for your scenario, e.g., opening drawer, placing objects...)

        except Exception as e:
            print(f"[Task] Exception during main task sequence: {e}")

        print("[Task] Main task execution complete.")

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()