# run_skeleton_task.py (Executable: Exploration for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only pre-existing skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Begin Exploration Phase: Try to identify which predicates exist and match the feedback.
        # The feedback provided is: (robot-at dice1_pose)
        # This suggests we should check if 'robot-at <location>' atom appears in the environment state.

        # STEP 1: List locations, objects, robot
        # We assume the positions dictionary keys give object names / locations
        # Let us try to find a location named 'dice1_pose' and see if robot moves there

        dice1_pose = None
        # Try to find the pose called dice1_pose or similar in positions
        for k in positions:
            if 'dice1_pose' in k:
                dice1_pose = k
                break
        if dice1_pose is None:
            # Fallback: try to find a location that may represent dice1_pose or print all available
            print("[Exploration] dice1_pose not found in get_object_positions(). Available keys:", list(positions.keys()))
            print("[Exploration] Unable to continue exploration without dice1_pose.")
            return

        # Determine robot's current position (if available)
        robot_loc = None
        for k in positions:
            if 'robot' in k.lower() or 'base' in k.lower():
                robot_loc = k
                break
        if robot_loc is None:
            # If not coded as such, assume a default start location (will need adjustment for true env)
            robot_loc = list(positions.keys())[0]

        # STEP 2: Use only allowed skill 'execute_go' to move the robot from its current location to dice1_pose
        try:
            print(f"[Exploration] Attempting to move robot from {robot_loc} to {dice1_pose} using execute_go.")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_loc,
                to_location=dice1_pose
            )
            print(f"[Exploration] Moved robot to {dice1_pose}.")
        except Exception as e:
            print(f"[Exploration] Error during execute_go: {e}")
            return

        # STEP 3: Check resulting observations for robot-at predicate
        # For illustration output, print relevant info to help debug the current state.
        try:
            new_obs = task.get_observation()
            print("[Exploration] Observation after move:", new_obs)

            # Optionally process observation if possible to check for 'robot-at dice1_pose'
        except Exception as e:
            print(f"[Exploration] Error retrieving observation after move: {e}")

        # STEP 4: Try execute_pull or other relevant actions to discover lock-known, if applicable
        # In this exploration, only allowed to use provided skills. So if there is a handle and drawer, etc.
        # For now, we stop exploration after checking robot-at since feedback is about this predicate.

        print("[Exploration] Exploration phase finished. Inspect output above to detect missing predicate.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()