# run_skeleton_task.py (Completed Executable for Predicate Exploration)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills, do not redefine

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Get object positions or environment states
        positions = get_object_positions()

        # --- Exploration Phase for Missing Predicate ---
        # The feedback states (drawer-open drawer2) is missing; thus, we need to explore to discover such predicates

        # Assumptions:
        # - There is a drawer2 entity in the environment (must appear in positions or env state).
        # - There is a handle attached to drawer2.
        # - The robot must explore drawers to determine the open/closed status.

        # For illustration, we assume drawer2, handles, and their positions are available in positions dict.
        # Replace key strings as appropriate—this code is robust to key errors & missing info.

        try:
            # Find all drawers and handles
            drawer_names = [name for name in positions.keys() if 'drawer' in name]
            handle_names = [name for name in positions.keys() if 'handle' in name]

            print(f"Found drawers: {drawer_names}")
            print(f"Found handles: {handle_names}")

            # Example: Let’s focus on 'drawer2'
            drawer_to_explore = None
            for d in drawer_names:
                if '2' in d:
                    drawer_to_explore = d
                    break
            if drawer_to_explore is None and drawer_names:
                drawer_to_explore = drawer_names[0]  # fallback

            print(f"Exploring drawer: {drawer_to_explore}")

            # Find handle for this drawer (if mapping available)
            handle_for_drawer = None
            for h in handle_names:
                if drawer_to_explore in h or h.replace('handle_', '') in drawer_to_explore:
                    handle_for_drawer = h
                    break
            if handle_for_drawer is None and handle_names:
                handle_for_drawer = handle_names[0]

            print(f"Using handle: {handle_for_drawer}")

            # ========== Exploration Sequence ==========

            # 1. Locate robot's current location
            robot_location = None
            for k, v in positions.items():
                if k.startswith('robot') or k == 'robot':
                    robot_location = v
                    break
            if robot_location is None:
                # If not given, pick a default location from locations list
                location_names = [name for name in positions.keys() if 'location' in name or 'drawer' in name]
                if location_names:
                    robot_location = positions[location_names[0]]
                else:
                    robot_location = (0,0,0)  # fallback

            print(f"Robot initial location: {robot_location}")

            # 2. Move robot to drawer location if not already there
            try:
                drawer_pos = positions[drawer_to_explore]
                if np.linalg.norm(np.array(robot_location) - np.array(drawer_pos)) > 0.05:
                    # Use execute_go to move
                    obs, reward, done = execute_go(env, task, from_pos=robot_location, to_pos=drawer_pos)
                    print(f"Moved to {drawer_to_explore} at {drawer_pos}")
                    robot_location = drawer_pos
            except Exception as e:
                print(f"Could not move to drawer due to: {e}")

            # 3. (Optional) Pick the handle to interact with drawer
            try:
                handle_pos = positions[handle_for_drawer]
                obs, reward, done = execute_pick(env, task, target_obj=handle_for_drawer, target_pos=handle_pos)
                print(f"Picked {handle_for_drawer} at {handle_pos}")
            except Exception as e:
                print(f"Could not pick handle due to: {e}")

            # 4. Attempt to pull the drawer open (explore lock-known/open/closed predicates)
            # According to exploration domain: execute_pull reveals lock-known predicate.
            # According to main domain: execute_pull opens the drawer if unlocked and closed and holding handle.

            try:
                obs, reward, done = execute_pull(env, task, drawer=drawer_to_explore, handle=handle_for_drawer, location=robot_location)
                print(f"Pulled to open {drawer_to_explore}")
            except Exception as e:
                print(f"Pull action failed (maybe drawer already open or locked): {e}")

            # 5. Check by perception/observation if the drawer is now open; reveal (drawer-open drawer2)
            print("[Exploration] Checking if drawer is open via sensor or observation.")

            # You could do:
            # obs_state = task.get_observation()
            # print(f"Observation after pull: {obs_state}")

            # 6. If the predicate (drawer-open drawer2) is revealed, exploration is successful.
            # Otherwise, repeat or try with another drawer.

        except Exception as main_ex:
            print(f"Exploration phase failed: {main_ex}")

        # ---------- END OF EXPLORATION ----------

        print("Exploration phase complete. Any missing predicates (like (drawer-open drawer2)) should now be discovered.")

        # Continue with further plan execution if needed, or exit.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
