# run_skeleton_task.py (Filled for Exploration Phase to Detect Missing Predicates)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

# This module gives object positions or other info about the environment
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()  # Dict: object_name -> pos tuple
        # Print positions for debugging
        print("[Info] Object positions:", positions)

        # --------------------------------------------------------------
        # EXPLORATION PHASE: Try all available skills to detect missing predicates!
        # When planner times out or preconditions not met, we need to explore environment,
        # e.g., checking predicates like 'weight-known', 'identified', etc.
        # --------------------------------------------------------------

        # Step 1: Gather all available skill names
        skills = ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 
                  'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # Assuming we can get object and drawer/location names (from positions and task descriptions)
        object_names = [name for name in positions.keys() if "drawer" not in name and "handle" not in name]
        drawer_names = [name for name in positions.keys() if "drawer" in name]
        handle_names = [name for name in positions.keys() if "handle" in name]
        location_names = []  # If positions gives locs, use those; otherwise infer
        for name in positions.keys():
            if "loc" in name or "location" in name or "room" in name:
                location_names.append(name)
        # Fallback: Use drawer names as locations (since robot moves to drawers)
        if not location_names:
            location_names = drawer_names.copy()

        # Try to execute each skill once on all relevant objects/locations, catch exceptions and
        # print meaningful errors to find missing predicates.
        # This can help the user/data scientist to realize which predicates are missing or required.

        # -- EXPLORATION: Try execute_go between two locations --
        if len(location_names) >= 2:
            from_loc, to_loc = location_names[0], location_names[1]
            print(f"[Explore] Trying execute_go from {from_loc} to {to_loc}")
            try:
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_loc,
                    to_loc
                )
                print(f"[Explore] execute_go succeeded: from {from_loc} to {to_loc}")
            except Exception as e:
                print(f"[Explore] execute_go failed: {e}")

        # -- EXPLORATION: Try execute_pick on each object at each location --
        for obj in object_names:
            for loc in location_names:
                print(f"[Explore] Trying execute_pick of {obj} at {loc}")
                try:
                    obs, reward, done = execute_pick(
                        env,
                        task,
                        obj,
                        loc
                    )
                    print(f"[Explore] execute_pick succeeded: {obj} at {loc}")
                except Exception as e:
                    print(f"[Explore] execute_pick failed for {obj}@{loc}: {e}")

        # -- EXPLORATION: Try execute_place for all (obj,drawer,loc) combinations --
        for obj in object_names:
            for d in drawer_names:
                for loc in location_names:
                    print(f"[Explore] Trying execute_place: {obj} into {d} at {loc}")
                    try:
                        obs, reward, done = execute_place(
                            env,
                            task,
                            obj,
                            d,
                            loc
                        )
                        print(f"[Explore] execute_place succeeded: {obj} -> {d} at {loc}")
                    except Exception as e:
                        print(f"[Explore] execute_place failed for {obj}->{d}@{loc}: {e}")

        # -- EXPLORATION: Try execute_pull on all drawers with all handles at all locations --
        for d in drawer_names:
            for h in handle_names:
                for loc in location_names:
                    print(f"[Explore] Trying execute_pull on {d} with handle {h} at {loc}")
                    try:
                        obs, reward, done = execute_pull(
                            env,
                            task,
                            d,
                            h,
                            loc
                        )
                        print(f"[Explore] execute_pull succeeded: {d}, handle {h} at {loc}")
                    except Exception as e:
                        print(f"[Explore] execute_pull failed for {d} handle {h} at {loc}: {e}")

        # -- EXPLORATION: Try execute_push on all drawers at all locations --
        for d in drawer_names:
            for loc in location_names:
                print(f"[Explore] Trying execute_push on {d} at {loc}")
                try:
                    obs, reward, done = execute_push(
                        env,
                        task,
                        d,
                        loc
                    )
                    print(f"[Explore] execute_push succeeded: {d} at {loc}")
                except Exception as e:
                    print(f"[Explore] execute_push failed for {d} at {loc}: {e}")

        # -- EXPLORATION: Try execute_sweep on each object at each location --
        for obj in object_names:
            for loc in location_names:
                print(f"[Explore] Trying execute_sweep of {obj} at {loc}")
                try:
                    obs, reward, done = execute_sweep(
                        env,
                        task,
                        obj,
                        loc
                    )
                    print(f"[Explore] execute_sweep succeeded: {obj} at {loc}")
                except Exception as e:
                    print(f"[Explore] execute_sweep failed for {obj} at {loc}: {e}")

        # -- EXPLORATION: Try execute_gripper with no params (robot-free) --
        print(f"[Explore] Trying execute_gripper()")
        try:
            obs, reward, done = execute_gripper(
                env,
                task
            )
            print(f"[Explore] execute_gripper succeeded")
        except Exception as e:
            print(f"[Explore] execute_gripper failed: {e}")

        print("===== Exploration phase complete. Analyze printed errors for missing predicates. =====")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
