# run_skeleton_task.py (Completed with Exploration for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use all provided skills directly

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Example structure: {'object1': (x, y, z), 'object2': (x, y, z), ...}

        # === 1. Exploration Phase: Attempt to Detect Missing Predicates ===
        #
        # When planning fails or solver times out, it's often due to missing or incomplete
        # knowledge in the PDDL (e.g. a predicate such as lock-known, temperature-known, etc.)
        # The purpose here is to perform "exploratory" actions using the available skills 
        # to actively uncover unknown predicates by manipulating and sensing objects/drawers.
        #
        # We use the available skills (execute_pick, execute_place, execute_pull, etc.)
        # to interact with the environment, and monitor feedback for possible missing information.

        # NOTE: Since the exact object and location names are not statically specified,
        # we'll iterate over all detected objects, optionally all drawers if present.

        # === Step 1: Try to open all drawers (possible missing lock-known predicate) ===
        # If drawer does not open, possibly due to missing lock knowledge, log it.

        # For demonstration, assume object type hinting in positions or descriptions.
        # Otherwise, fallback to names (e.g., any object with "drawer" in name treated as drawer).
        detected_drawers = [obj for obj in positions if "drawer" in obj.lower()]
        detected_handles = [obj for obj in positions if "handle" in obj.lower()]
        detected_objects = [obj for obj in positions if obj not in detected_drawers and obj not in detected_handles]

        # For this demonstration, suppose we have at least one drawer and its handle
        for drawer_name in detected_drawers:
            try:
                drawer_pos = positions[drawer_name]
                # Find associated handle
                handle_name = None
                for h in detected_handles:
                    if drawer_name in h:
                        handle_name = h
                        break
                if handle_name is None and len(detected_handles) > 0:
                    handle_name = detected_handles[0]  # fallback

                # Move robot to the drawer's location
                print(f"[Exploration] Approaching {drawer_name}")
                obs, reward, done = execute_go(
                    env, 
                    task,
                    from_location=None,  # Assume env/task can resolve current location
                    to_location=drawer_pos
                )
                
                if done:
                    print(f"[Exploration] Unexpected task completion after move to {drawer_name}")
                    break

                # Try to pick the handle if found
                if handle_name is not None:
                    handle_pos = positions[handle_name]
                    print(f"[Exploration] Trying to pick {handle_name} at {handle_pos}")
                    try:
                        obs, reward, done = execute_pick(
                            env,
                            task,
                            obj=handle_name,
                            location=drawer_pos
                        )
                    except Exception as exc:
                        print(f"[Exploration] Failed to pick handle {handle_name}: {exc}")
                        continue
                else:
                    print(f"[Exploration] No handle found for {drawer_name}, skipping pull test.")
                    continue

                # Try to pull the drawer using the handle
                print(f"[Exploration] Attempting to pull {drawer_name} using {handle_name}.")
                try:
                    obs, reward, done = execute_pull(
                        env,
                        task,
                        drawer_name,
                        handle_name,
                        drawer_pos
                    )
                except Exception as exc:
                    print(f"[Exploration] Pull action failed for drawer {drawer_name}: {exc}")
                    print("  Note: If this repeatedly fails, it may indicate that a missing predicate (e.g., lock-known) or insufficient knowledge about lock states exists in the PDDL domain/problem.")
                    continue

            except Exception as e:
                print(f"[Exploration] Exception during drawer exploration: {e}")
                continue

        # === Step 2: Try picking and placing all known objects (missing knowledge: weight, durability, etc.) ===

        for obj_name in detected_objects:
            try:
                obj_pos = positions[obj_name]
                print(f"[Exploration] Trying to pick object {obj_name} at {obj_pos}")
                try:
                    obs, reward, done = execute_pick(
                        env,
                        task,
                        obj=obj_name,
                        location=obj_pos
                    )
                except Exception as exc:
                    print(f"[Exploration] Could not pick {obj_name}: {exc}")
                    continue

                # Try placing in each available drawer (if drawers exist)
                for dname in detected_drawers:
                    dpos = positions[dname]
                    print(f"[Exploration] Trying to place {obj_name} in drawer {dname} at {dpos}")
                    try:
                        obs, reward, done = execute_place(
                            env,
                            task,
                            obj=obj_name,
                            drawer=dname,
                            location=dpos
                        )
                    except Exception as exc:
                        print(f"[Exploration] Could not place {obj_name} in {dname}: {exc}")
                        continue

            except Exception as single_obj_exc:
                print(f"[Exploration] Error with object {obj_name}: {single_obj_exc}")

        print("===== Exploration Phase Complete =====")
        print("If persistent action/plan failures remain, PDDL domain may lack predicates like lock-known, temperature-known, or similar. Refer to exploration domain and planner output for guidance.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
