# run_skeleton_task.py

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # use only *provided* primitive skill implementations

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = 'object_1': (0,0,0), 'object_2': (1,1,1), ... (dictionary format)
        positions = get_object_positions()

        # --- Exploration and Predicate Identification Phase ---
        print("[Exploration] Attempt to pull drawer1 (identify lock status)...")

        # 1. Move robot to the drawer location (assume robot starts not at drawer)
        try:
            drawer_loc = positions.get('drawer1')
            if drawer_loc is None:
                raise Exception("drawer1 location not found in object positions.")
            robot_start = positions.get('robot', positions.get('robot_start', None))
            if robot_start and robot_start != drawer_loc:
                # Move via "execute_go" primitive
                print(f"[Action] Moving robot from {robot_start} to {drawer_loc} via execute_go.")
                execute_go(env, robot_start, drawer_loc)
            else:
                print(f"[Info] Robot already at drawer1 location {drawer_loc}.")
        except Exception as e:
            print(f"[Warning] Could not execute_go to drawer1: {str(e)}")

        # 2. Try to pick the handle (assume handle is object 'handle1' and co-located with drawer1)
        handle_name = 'handle1'
        try:
            handle_pos = positions.get(handle_name)
            if handle_pos is None:
                raise Exception(f"{handle_name} position not found.")
            print(f"[Action] Attempting execute_pick on {handle_name} at {drawer_loc}.")
            execute_pick(env, handle_name, drawer_loc)
        except Exception as e:
            print(f"[Warning] Could not execute_pick {handle_name}: {str(e)}")

        # 3. Try to pull the drawer open (exploration for lock)
        try:
            print(f"[Action] Attempting execute_pull on drawer1 with {handle_name} at {drawer_loc}.")
            # In the pure domain, the execute_pull needs (drawer, handle_object, location)
            execute_pull(env, 'drawer1', handle_name, drawer_loc)
        except Exception as e:
            print(f"[Feedback] execute_pull failed for drawer1, indicating a possible missing predicate or lock state: {e}")

        # At this point, you have gathered feedback for the missing predicate (e.g., (drawer-closed drawer1) or lock known)

        # --- Main Task Execution ---

        # You would now proceed with your actual task plan using predefined skill names.
        # Below is an example template you can adapt when you have the oracle plan.

        # ---- Example plan execution for opening a locked drawer and putting an object inside ----
        # 0. ASSUMPTIONS
        #  - robot starts not holding anything, at initial (known) location
        #  - 'obj1' to be picked up from floor and placed in drawer1
        #  - drawer1 is initially locked and closed; handle1 is on the drawer

        object_name = 'obj1'
        place_in_drawer = 'drawer1'
        place_location = drawer_loc  # likely same as drawer

        try:
            # 4. (If needed) Move to object to pick up (skipped if already at drawer)
            obj_pos = positions.get(object_name)
            robot_loc = positions.get('robot', positions.get('robot_start', None))

            if obj_pos is not None and robot_loc != obj_pos:
                print(f"[Action] Moving robot from {robot_loc} to {obj_pos} to pick up {object_name}.")
                execute_go(env, robot_loc, obj_pos)
            else:
                print(f"[Info] Robot already at {object_name}.")
        except Exception as e:
            print(f"[Warning] Could not execute_go to {object_name}: {str(e)}")

        # 5. Pick up object from floor
        try:
            print(f"[Action] Picking up {object_name} at {obj_pos} using execute_pick.")
            execute_pick(env, object_name, obj_pos)
        except Exception as e:
            print(f"[Warning] Could not execute_pick {object_name}: {str(e)}")

        # 6. Move to drawer location (if needed)
        try:
            if obj_pos != drawer_loc:
                print(f"[Action] Moving robot with {object_name} to {drawer_loc} using execute_go.")
                execute_go(env, obj_pos, drawer_loc)
        except Exception as e:
            print(f"[Warning] Could not execute_go to {drawer_loc}: {str(e)}")

        # 7. Open drawer (assume lock/unlock has been handled outside or is not needed)
        # Must pick handle and pull (as above)
        try:
            print(f"[Action] Attempting execute_pick on handle {handle_name} for drawer opening.")
            execute_pick(env, handle_name, drawer_loc)
        except Exception as e:
            print(f"[Warning] Could not execute_pick {handle_name} for drawer: {str(e)}")
        try:
            print(f"[Action] Attempting execute_pull for opening the drawer.")
            execute_pull(env, place_in_drawer, handle_name, drawer_loc)
        except Exception as e:
            print(f"[Warning] Could not execute_pull for drawer: {str(e)}")

        # 8. Place object inside drawer
        try:
            print(f"[Action] Placing {object_name} inside {place_in_drawer} (drawer) at {drawer_loc}.")
            execute_place(env, object_name, place_in_drawer, drawer_loc)
        except Exception as e:
            print(f"[Warning] Could not execute_place {object_name}: {str(e)}")

        # 9. Push the drawer closed (optional, depending on goal)
        try:
            print(f"[Action] Attempting execute_push to close the drawer.")
            execute_push(env, place_in_drawer, drawer_loc)
        except Exception as e:
            print(f"[Warning] Could not execute_push for drawer: {str(e)}")

        print("[Task] Oracle plan executed (with exploration for missing predicates).")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()