# run_skeleton_task.py (Completed Executable Code with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Predefined skill functions must be used as-is

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with missing predicate exploration.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers for recording
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Get object positions mapping
        positions = get_object_positions()
        # e.g., positions might return {'robot': (x,y,z), 'drawer1': (x,y,z), 'obj1': (x,y,z), ...}

        # --- EXPLORATION: Try to find missing predicates (drawer-unlocked) ---

        # Assume domain hints that 'drawer-unlocked' is required but not directly observable.
        # We'll try a sequence of skills to discover/trigger feedback for missing predicate.

        # Step 1: Move robot near drawer1 (if not already there)
        # We'll attempt to go to the location of drawer1
        robot_location = None
        drawer1_location = None

        for key in positions:
            if 'robot' in key:
                robot_location = key
            if 'drawer1' in key:
                drawer1_location = key

        if robot_location is None or drawer1_location is None:
            print("[Error] Unable to find required robot/drawer1 in positions dict.")
            return

        if robot_location != drawer1_location:
            print(f"[Exploration] Moving robot from {robot_location} to {drawer1_location} to approach drawer1.")
            try:
                obs, reward, done, info = execute_go(env, task, from_location=robot_location, to_location=drawer1_location)
            except Exception as e:
                print(f"[Error] Execute_go failed during exploration: {e}")
                return
            robot_location = drawer1_location

        # Step 2: Attempt to pull drawer1 using its handle (simulate to see if drawer can be opened)
        # We need to know the handle (object) for drawer1
        handle_obj_name = None
        for key in positions:
            if 'handle' in key and 'drawer1' in key:
                handle_obj_name = key
                break
        # Fallback if not found
        if handle_obj_name is None:
            # Try a generic handle or make an assumption
            for key in positions:
                if 'handle' in key:
                    handle_obj_name = key
                    break

        if handle_obj_name is None:
            print("[Error] No handle object found for drawer1, cannot attempt execute_pull.")
            return

        # Ensure robot's hand is empty and robot-free before pick
        print(f"[Exploration] Picking up handle {handle_obj_name} to pull drawer1.")
        try:
            obs, reward, done, info = execute_pick(env, task, o=handle_obj_name, p=drawer1_location)
        except Exception as e:
            print(f"[Error] Failed to pick handle during exploration: {e}")
            return

        print("[Exploration] Attempting to execute_pull (open drawer1 via handle).")
        try:
            obs, reward, done, info = execute_pull(env, task, d='drawer1', h=handle_obj_name, p=drawer1_location)
        except Exception as e:
            print("[Exploration] execute_pull action failed (likely due to missing predicate such as drawer-unlocked).")
            print(f"[Feedback] Exploration confirmed missing or unmet predicate: (drawer-unlocked drawer1) needed to pull/open drawer!")
            # At this point, we've triggered the missing predicate feedback and can now code the rest conditionally.
            print("[Exploration] Exploration phase complete. Proceed to plan implementation after learning missing predicate.")
            return

        print("[Exploration] Drawer opened without missing predicate problem (unexpected). Exploration ends.")

        # === PLAN EXECUTION PHASE ===
        # Now that the exploration feedback made clear (drawer-unlocked drawer1) is required, 
        # the normal oracle plan should only attempt pull after confirming/unlocking this predicate.

        # Example skeleton steps (filling in as if predicates are now satisfied):
        # - (Optional) unlock drawer1 (if a skill for unlocking exists, use it)
        # - Move (execute_go) robot near objects/locations as needed
        # - Pick object(s) (execute_pick)
        # - Place object(s) in drawer (execute_place)
        # - Push/close drawer (execute_push)

        # Example Oracle Plan (fictitious, as no specific plan provided):

        # 1. Move robot to object location (assume 'obj1' to store in 'drawer1')
        obj1_location = None
        for key in positions:
            if 'obj1' in key:
                obj1_location = key
                break
        if obj1_location is None:
            print("[Error] No obj1 found in environment for plan execution.")
            return

        # Assume robot is no longer at object location, so move there if needed
        if robot_location != obj1_location:
            try:
                print(f"[Plan] Moving robot from {robot_location} to {obj1_location} for pick.")
                obs, reward, done, info = execute_go(env, task, from_location=robot_location, to_location=obj1_location)
                robot_location = obj1_location
            except Exception as e:
                print(f"[Error] execute_go failed during plan: {e}")
                return

        # 2. Pick up the object (obj1)
        try:
            print(f"[Plan] Picking up object {obj1_location} (obj1).")
            obs, reward, done, info = execute_pick(env, task, o='obj1', p=obj1_location)
        except Exception as e:
            print(f"[Error] execute_pick failed: {e}")
            return

        # 3. Move to drawer1
        if robot_location != drawer1_location:
            try:
                print(f"[Plan] Moving robot from {robot_location} to {drawer1_location} to place object.")
                obs, reward, done, info = execute_go(env, task, from_location=robot_location, to_location=drawer1_location)
                robot_location = drawer1_location
            except Exception as e:
                print(f"[Error] execute_go failed: {e}")
                return

        # 4. Place object in drawer1 (assuming the drawer is now open OR
        # after unlocking/opening per the domain if necessary)
        try:
            print("[Plan] Placing obj1 into drawer1 at", drawer1_location)
            obs, reward, done, info = execute_place(env, task, o='obj1', d='drawer1', p=drawer1_location)
        except Exception as e:
            print(f"[Error] execute_place failed: {e}")
            return

        # 5. Close (push) drawer1
        try:
            print("[Plan] Closing drawer1 by push.")
            obs, reward, done, info = execute_push(env, task, d='drawer1', p=drawer1_location)
        except Exception as e:
            print(f"[Error] execute_push failed: {e}")
            return

        print("[Plan] Task completed successfully.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()