# run_skeleton_task.py (Completed for Exploration and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only the predefined skill functions

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions/State Information ===
        positions = get_object_positions()

        # ------ Exploration Phase: Identifying Missing Predicate ------
        # Based on the feedback: (drawer-closed drawer1)
        # We need to explicitly check and ensure the predicate 'drawer-closed' or any similar required state is detectable in simulation.

        # We'll proceed to try to explore or manipulate the drawer to detect its state.
        # This includes using available skills to operate on the drawer, and observing effects.

        # You may need to change these names based on your objects/environment:
        drawer_name = 'drawer1'
        handle_name = None
        # Find the handle associated with drawer1 (if that info is available)
        for obj in positions:
            if 'handle' in obj and drawer_name in obj:
                handle_name = obj
                break
        if handle_name is None:
            # Fallback: try to find any handle object in positions
            for obj in positions:
                if 'handle' in obj:
                    handle_name = obj
                    break

        # Assume robot and drawer are at same location for the experiment. If not, move the robot.
        robot_location = None
        drawer_location = None
        for obj, pos in positions.items():
            if obj == 'robot':
                robot_location = pos
            if obj == drawer_name:
                drawer_location = pos

        # If locations are names:
        robot_at = None
        drawer_at = None
        for obj, pos in positions.items():
            if obj == 'robot':
                robot_at = pos if isinstance(pos, str) else None
            if obj == drawer_name:
                drawer_at = pos if isinstance(pos, str) else None

        # === Exploration: Move robot to drawer location if needed ===
        try:
            if robot_at and drawer_at and robot_at != drawer_at:
                # Use execute_go (predefined skill) to move the robot
                print(f"[Exploration] Moving robot from {robot_at} to {drawer_at}")
                obs, reward, done = execute_go(env, task, robot_at, drawer_at)
                if done:
                    print("[Exploration] Unable to move robot to drawer; exploration failed.")
                    return
                robot_at = drawer_at
        except Exception as e:
            print(f"[Exploration] Error during robot movement: {e}")

        # === Exploration: Try to Pull (Open) Drawer to Check Status ===
        # To check if 'drawer-closed' is the missing predicate required for opening,
        # we will attempt to pick the handle and then pull the drawer.
        try:
            # Pick the handle first if not holding it.
            if handle_name:
                print(f"[Exploration] Attempting to pick handle: {handle_name}")
                obs, reward, done = execute_pick(env, task, handle_name, drawer_at)
                if done:
                    print("[Exploration] Failed to pick handle; cannot explore further.")
                    return
                # Now pull the drawer (try to open). This will only work if drawer is unlocked and closed.
                print(f"[Exploration] Attempting to pull (open) drawer: {drawer_name} using handle: {handle_name}")
                obs, reward, done = execute_pull(env, task, drawer_name, handle_name, drawer_at)
                if done:
                    print("[Exploration] Drawer may not be closed, locked, or other preconditions not met.")
                else:
                    print("[Exploration] Pull action succeeded. Drawer was likely closed and is now open.")
            else:
                print("[Exploration] Could not find a suitable handle for the drawer.")

        except Exception as e:
            print(f"[Exploration] Exception during pull testing: {e}")

        # === (Optional) Check other effects or record state ===
        # At this stage, we know that interacting with the drawer required checking the 'drawer-closed' predicate,
        # as per the feedback. The exploration should have confirmed or surfaced the predicate situation.

        # === Task Plan Execution - Example Sequence ===
        # --- Replace this with the actual oracle plan actions as per your experiment or next steps. ---
        #
        # For demonstration, attempt to close the drawer after the pull (open):
        #
        try:
            print(f"[Task] Attempting to push (close) the drawer: {drawer_name} at {drawer_at}")
            obs, reward, done = execute_push(env, task, drawer_name, drawer_at)
            if done:
                print("[Task] Drawer push (close) succeeded.")
            else:
                print("[Task] Drawer push (close) did not complete the task; further actions may be required.")
        except Exception as e:
            print(f"[Task] Exception during drawer push: {e}")

        # At this point, the exploration and manipulation should have either clarified requirements or advanced the state as needed.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
