# run_skeleton_task.py (Executable Exploration/Skeleton)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use all existing skill functions; do not redefine.

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)
        positions = get_object_positions()
        
        # === Identify objects, locations, drawers, handles ===
        # Attempt to infer key objects from object_positions dictionary
        object_names = [k for k in positions.keys() if "obj" in k or "object" in k or "ball" in k]
        drawer_names = [k for k in positions.keys() if "drawer" in k]
        handle_names = [k for k in positions.keys() if "handle" in k]
        location_names = [k for k in positions.keys() if "loc" in k or "table" in k or "room" in k]

        # To track current robot state and missing predicates
        missing_predicate_found = False
        missing_predicate_name = None

        # === Exploration Phase: Determine Which Predicate is Missing ===

        # Attempt to perform all available skills (that are safe for exploration)
        # Focus on those that involve robot state, e.g. whether (robot-free) is required
        # Feedback suggested (robot-free) predicate was missing/or issue was found

        # 1. Try to pick each object on the floor at each location:
        print("[Exploration] Checking robot-free and hand-empty through pick actions ...")
        for obj_name in object_names:
            for loc_name in location_names:
                try:
                    # Attempt to call execute_pick with each object/location
                    # Skill signature: execute_pick(env, task, object_name, location_name, ...)
                    obs, reward, done = execute_pick(
                        env,
                        task,
                        target_object=obj_name,
                        location=loc_name,
                        approach_distance=0.1,
                        max_steps=80,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    print(f"[Exploration] Pick executed for {obj_name} at {loc_name}, result: {done}")
                    if not done:
                        # If picking failed, maybe due to missing precondition, note it
                        missing_predicate_name = "robot-free or hand-empty"
                except Exception as e:
                    print(f"[Exploration] Exception during pick: {e}")
                    missing_predicate_name = "robot-free or hand-empty"
                    continue

        # 2. Try to pull a drawer using handles, with the expected preconditions
        print("[Exploration] Checking drawer manipulation + predicates ...")
        for drawer_name in drawer_names:
            for handle_name in handle_names:
                for loc_name in location_names:
                    try:
                        # Need to pick handle first before pulling
                        print(f"[Exploration] Attempt pick (handle) before pull: {handle_name} at {loc_name}")
                        obs, reward, done = execute_pick(
                            env,
                            task,
                            target_object=handle_name,
                            location=loc_name,
                            approach_distance=0.1,
                            max_steps=80,
                            threshold=0.01,
                            approach_axis='z',
                            timeout=10.0
                        )
                        if done:
                            print(f"[Exploration] Picked handle {handle_name}, now executing pull on {drawer_name}")
                            obs, reward, done = execute_pull(
                                env,
                                task,
                                drawer=drawer_name,
                                handle=handle_name,
                                location=loc_name,
                                max_steps=100
                            )
                            print(f"[Exploration] Pull executed for {drawer_name} via {handle_name} at {loc_name}, result: {done}")
                    except Exception as e:
                        print(f"[Exploration] Exception during pull: {e}")
                        missing_predicate_name = "drawer-locked/drawer-unlocked or robot-free"
                        continue

        # 3. Try to push drawers closed (robot-free precondition)
        print("[Exploration] Checking drawer push + predicates ...")
        for drawer_name in drawer_names:
            for loc_name in location_names:
                try:
                    obs, reward, done = execute_push(
                        env,
                        task,
                        drawer=drawer_name,
                        location=loc_name,
                        max_steps=80
                    )
                    print(f"[Exploration] Push executed for {drawer_name} at {loc_name}, result: {done}")
                except Exception as e:
                    print(f"[Exploration] Exception during push: {e}")
                    missing_predicate_name = "robot-free"
                    continue

        # 4. Try to move (execute_go) between locations
        print("[Exploration] Checking go (move) skill + predicates ...")
        for i, from_loc in enumerate(location_names):
            for j, to_loc in enumerate(location_names):
                if i != j:
                    try:
                        obs, reward, done = execute_go(
                            env,
                            task,
                            from_location=from_loc,
                            to_location=to_loc,
                            max_steps=120
                        )
                        print(f"[Exploration] Move executed from {from_loc} to {to_loc}, result: {done}")
                    except Exception as e:
                        print(f"[Exploration] Exception during go/move: {e}")
                        missing_predicate_name = "robot-free"
                        continue

        # After exploration, if we've observed a missing precondition or error, log it
        if missing_predicate_name:
            print(f"\n[Exploration Result] Likely missing predicate: {missing_predicate_name}\n")
        else:
            print("\n[Exploration Result] No clearly missing predicate detected in skill execution.\n")

        # === If exploration reveals missing predicate, attempt to ensure robot-free before actions ===
        # (In a true system, plan logic would insert a wait or hand-empty or robot-free skill, but we just report.)

        print("[Task] Exploration complete. Now ready for actual plan execution / further logic as needed.")

    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()