# run_skeleton_task.py (Executable Exploration-Aware Task)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # use only provided skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        
        # At this point, we should have all provided object and location positions as a dict
        # For example: positions = {'robot': ..., 'drawer-area': ..., 'objectA': ..., ... }

        # === Feedback Exploration: Check for Predicate Presence ===
        # Feedback: missing predicate (robot-at drawer-area)
        # The predicate (robot-at drawer-area) is expected in the planning state but may not be correctly
        # handled unless exploration ensures the location is discovered/confirmed.

        # Step 1: Exploration phase to ensure knowledge about 'drawer-area' presence.
        # (If relevant, try to move the robot to drawer-area and confirm.)

        drawer_area_pos = None
        robot_pos = None
        for key in positions:
            if 'drawer-area' in key:
                drawer_area_pos = positions[key]
            if 'robot' in key:
                robot_pos = positions[key]
        if drawer_area_pos is None:
            print("[Exploration] Warning: Couldn't find 'drawer-area' position in object positions!")
        if robot_pos is None:
            print("[Exploration] Warning: Couldn't find 'robot' position in object positions!")
        
        # Plan: Move the robot to the drawer-area and confirm presence (to identify (robot-at drawer-area))
        # Use only predefined skills: execute_go
        try:
            if drawer_area_pos is not None and robot_pos is not None:
                print("[Exploration] Moving robot to drawer-area for predicate confirmation/exploration.")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_pos=robot_pos,
                    to_pos=drawer_area_pos,
                    max_steps=100,
                    threshold=0.01,
                    timeout=5.0
                )
                # Update local robot position
                robot_pos = drawer_area_pos
                if done:
                    print("[Exploration] Reached drawer-area.")
            else:
                print("[Exploration] Skipping move due to missing positions.")

        except Exception as e:
            print("[Exploration] Exception during execute_go:", str(e))

        # Now (robot-at drawer-area) should be true in the environment model
        # (If this predicate is critical for follow-up plans, we have ensured it.)

        # === Resume Main Plan Logic Here ===
        # -- This is the point where you'd use other skill actions as per your oracle plan --
        # For demonstration, let's assume some main task after exploration (just as placeholder):

        # Example placeholder for further skills (adjust as needed for your actual task/plan):
        # (Remove if you have a fixed plan sequence to execute)

        # e.g., pick up a handle after moving to drawer-area
        handle_obj = None
        for k in positions:
            if 'handle' in k:
                handle_obj = k
                break
        if handle_obj and drawer_area_pos is not None:
            try:
                print("[Task] Attempting to pick handle at drawer-area.")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    obj_name=handle_obj,
                    location=drawer_area_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print("[Task] Picked up handle.")
            except Exception as e:
                print("[Task] Exception during execute_pick for handle:", str(e))
        else:
            print("[Task] Handle object or drawer-area missing; skipping pick.")

        # e.g., execute_pull (if handle is held and drawer present)
        drawer_obj = None
        for k in positions:
            if 'drawer' in k and 'area' not in k:
                drawer_obj = k
                break
        if drawer_obj and handle_obj and drawer_area_pos is not None:
            try:
                print("[Task] Attempting to pull (open) the drawer using handle.")
                obs, reward, done = execute_pull(
                    env,
                    task,
                    drawer=drawer_obj,
                    handle=handle_obj,
                    location=drawer_area_pos,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                if done:
                    print("[Task] Pulled/opened the drawer.")
            except Exception as e:
                print("[Task] Exception during execute_pull:", str(e))
        else:
            print("[Task] Drawer object/handle/drawer-area missing; skipping pull.")

        # --- ADDITIONAL PLAN LOGIC AS NEEDED FOR YOUR ORACLE PLAN HERE ---

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
