# run_skeleton_task.py (Completed with Exploration Phase for Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only exported primitives

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # The feedback reveals a predicate: (drawer-closed drawer1)
        # Exploration phase: Check for the 'drawer-closed' property via available actions
        
        # --- EXPLORATION PHASE: Determine 'drawer-closed' status ---
        # For this, try executing an action using 'drawer1' that requires 'drawer-closed' as a precondition
        # As in domain, execute_pull requires (drawer-closed ?d)
        # Try to observe if the drawer1 is closed by attempting execute_pull
        # This will help confirm/identify the predicate through execution

        # This phase can be adapted to log the state or check action effects

        # Fetch object names (user must match them to their env, here we assume standard names)
        drawer_name = 'drawer1'
        handle_name = None
        robot_location = None

        # Find handle of drawer1 (assume positions dict or env provides such info)
        # In a real system, would query from env; here we attempt a direct mapping
        try:
            for obj_name, pos in positions.items():
                # Look for an object that is a handle of drawer1 (e.g., "handle1")
                if ('handle' in obj_name or 'Handle' in obj_name) and drawer_name in obj_name:
                    handle_name = obj_name
                    break
            if handle_name is None:
                # Fallback: just pick first handle
                for obj_name in positions:
                    if 'handle' in obj_name.lower():
                        handle_name = obj_name
                        break
        except Exception as e:
            print(f"[Exploration] Could not identify handle: {e}")

        # Find a robot location (arbitrarily pick the first location)
        try:
            for obj_name, pos in positions.items():
                if 'location' in obj_name.lower() or 'loc' in obj_name.lower():
                    robot_location = obj_name
                    break
            if robot_location is None:
                robot_location = 'location1'
        except Exception as e:
            robot_location = 'location1'

        # For exploration, attempt to pick the handle and pull, to test for 'drawer-closed'
        # These actions correspond to execute_pick (for handle), execute_pull (for drawer) -- per domain

        print("[Exploration] Attempting to pick handle and pull drawer to test drawer-closed state.")

        # --- 1. Go to the required location if necessary (per domain, execute_go) ---
        # Assume positions has robot start position; in real system you'd track robot state
        robot_pos = None
        try:
            for obj_name, pos in positions.items():
                if 'robot' in obj_name.lower():
                    robot_pos = obj_name
                    break
            if robot_pos is not None and robot_pos != robot_location:
                try:
                    obs, reward, done = execute_go(env, task, robot_pos, robot_location)
                except Exception as e:
                    print(f"[Exploration] Exception during execute_go: {e}")
        except Exception as e:
            pass

        # --- 2. Pick up the handle (requires hand empty, handle on floor, robot at location) ---
        try:
            obs, reward, done = execute_pick(env, task, handle_name, robot_location)
            print(f"[Exploration] Picked up handle {handle_name} at {robot_location}.")
        except Exception as e:
            print(f"[Exploration] Failed to pick up handle ({handle_name}): {e}")

        # --- 3. Try to pull the drawer (assessing precondition: drawer-closed, drawer-unlocked, etc.) ---
        # This corresponds to the action:
        # execute_pull(env, task, drawer, handle, location)
        try:
            obs, reward, done = execute_pull(env, task, drawer_name, handle_name, robot_location)
            print(f"[Exploration] Pulled on drawer {drawer_name} using handle {handle_name} at {robot_location}.")
        except Exception as e:
            print(f"[Exploration] Failed to pull drawer ({drawer_name}): {e}")
            print("[Exploration] This may indicate the predicate (drawer-closed drawer1) or some other precondition was not satisfied.")
            # Here, one would deduce the necessary predicate from the failed precondition.

        # At this point, exploration has identified the importance of the (drawer-closed drawer1) predicate.
        # --- END EXPLORATION PHASE ---

        # === (TASK PHASE) Example oracle plan execution ===
        # NOTE: In practice, the plan would be provided/generated at runtime,
        # here this is an illustration integrating available skills.

        # (1) Move to drawer location (if not already)
        # (2) Pick up handle
        # (3) Pull to open drawer (precondition: drawer-closed, drawer-unlocked)
        # (4) Place an object in the drawer, if desired (requires drawer-open, not full)
        # (5) Close drawer if policy requires

        # Example sequence using available skills (see available_skill_names)

        # 1. Go to drawer location
        try:
            obs, reward, done = execute_go(env, task, robot_location, robot_location)
        except Exception as e:
            print(f"[Task] Exception during execute_go: {e}")

        # 2. Pick up handle (may have done during exploration already, try again if needed)
        try:
            obs, reward, done = execute_pick(env, task, handle_name, robot_location)
        except Exception as e:
            print(f"[Task] Exception during execute_pick: {e}")

        # 3. Pull to open drawer
        try:
            obs, reward, done = execute_pull(env, task, drawer_name, handle_name, robot_location)
        except Exception as e:
            print(f"[Task] Exception during execute_pull: {e}")

        # 4. Place an object in the drawer (assume obj_name is known and at hand)
        # Find an object on the floor that is not a handle or drawer
        place_obj_name = None
        try:
            for name in positions.keys():
                if (
                    name != handle_name
                    and name != drawer_name
                    and "handle" not in name.lower()
                    and "drawer" not in name.lower()
                ):
                    place_obj_name = name
                    break
            if place_obj_name is not None:
                try:
                    # Pick up the object
                    obs, reward, done = execute_pick(env, task, place_obj_name, robot_location)
                except Exception as e:
                    print(f"[Task] Exception during execute_pick (place object): {e}")
                # Place it in the drawer
                try:
                    obs, reward, done = execute_place(env, task, place_obj_name, drawer_name, robot_location)
                except Exception as e:
                    print(f"[Task] Exception during execute_place: {e}")
            else:
                print("[Task] No available object to place in drawer.")
        except Exception as e:
            print(f"[Task] Error selecting object to place: {e}")

        # 5. Optionally, close the drawer
        try:
            obs, reward, done = execute_push(env, task, drawer_name, robot_location)
        except Exception as e:
            print(f"[Task] Exception during execute_push: {e}")

        print("===== Task logic (including exploration) complete =====")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
