# run_skeleton_task.py (Exploration-Enabled Skeleton)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use pre-defined primitive skills only

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Skeleton for running task & predicate exploration in simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions & State Info ===
        try:
            positions = get_object_positions()
        except Exception as e:
            print("[Error] Unable to get object positions:", e)
            positions = {}

        # === Exploration Phase: Try to Detect Missing Predicate ===
        #
        # The plan execution has previously failed to complete; this suggests a precondition or predicate required
        # for successful task completion is missing from the domain or initial state.
        #
        # To determine the missing predicate, try available exploration skills related to knowledge acquisition:
        # For this, attempt to execute all available exploration skills (e.g., object identification, temperature/weight/durability/lock sensing).
        # This step explores which information is missing and attempts to acquire it, thereby "unlocking" further plan progress.

        # -------------- Exploration Skill Mapping -------------------
        # Available: ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']
        # As per exploration domain, the mapping for knowledge discovery is (not all may be implemented):
        # - For location: execute_go (move to locations and observe)
        # - For weight: execute_pick (if supports knowledge gain)
        # - For lock: execute_pull (if drawer/handle can be tried)
        # Add any further exploration step logic here

        # Gather lists of drawers, objects, handles, locations from the positions dict
        objects = []
        drawers = []
        handles = []
        locations = set()
        for k, v in positions.items():
            if 'drawer' in k:
                drawers.append(k)
            elif 'handle' in k:
                handles.append(k)
            else:
                objects.append(k)
            # Every object is at a position; map to a location name
            if isinstance(v, dict) and 'location' in v:
                locations.add(v['location'])
            elif isinstance(v, str):
                locations.add(v)
        locations = list(locations)
        
        # Get robot/agent initial position if possible
        robot_loc = None
        if 'robot' in positions:
            robot_loc = positions.get('robot')
        elif 'robot_at' in positions:
            robot_loc = positions.get('robot_at')
        elif 'agent' in positions:
            robot_loc = positions.get('agent')
        else:
            # Try from observation or choose the first known location
            if locations:
                robot_loc = locations[0]
        
        # For the sake of demonstration, assume we can access available locations
        # and that object_locations is a mapping: object -> location

        missing_detected = False
        print(f"[Exploration] Starting knowledge exploration phase.")
        # Try to systematically attempt each available knowledge skill using available objects/locations
        
        # 1. Try to move to every location (simulate "execute_go") to trigger possible knowledge acquisition predicates
        for loc in locations:
            if robot_loc is None or robot_loc == loc:
                continue
            try:
                print(f"[Exploration] Attempting to move (execute_go) from '{robot_loc}' to '{loc}'.")
                obs, reward, done = execute_go(
                    env,
                    task,
                    start=robot_loc,
                    goal=loc,
                    max_steps=100,
                    threshold=0.01
                )
                print(f"[Exploration] Arrived at '{loc}'.")
                robot_loc = loc
            except Exception as e:
                print(f"[Exploration] Movement to {loc} failed:", e)
        
        # 2. Try picking each object at their positions if "execute_pick" available, which may yield new knowledge.
        for obj in objects:
            obj_info = positions.get(obj, {})
            # Determine the location of the object
            if isinstance(obj_info, dict) and 'location' in obj_info:
                obj_loc = obj_info['location']
            elif isinstance(obj_info, str):
                obj_loc = obj_info
            else:
                obj_loc = None
            # If robot's not at obj's location, attempt to move there first
            if obj_loc is not None and robot_loc != obj_loc:
                try:
                    print(f"[Exploration] Moving to '{obj_loc}' to try picking '{obj}'.")
                    obs, reward, done = execute_go(
                        env,
                        task,
                        start=robot_loc,
                        goal=obj_loc,
                        max_steps=100,
                        threshold=0.01
                    )
                    robot_loc = obj_loc
                except Exception as e:
                    print(f"[Exploration] Can't move to {obj_loc} for '{obj}':", e)
                    continue
            # Try pick action
            try:
                print(f"[Exploration] Trying to pick '{obj}' at '{obj_loc}'.")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    target_object=obj,
                    target_pos=obj_loc,
                    max_steps=100,
                    approach_distance=0.15,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                print(f"[Exploration] Picked '{obj}'.")
            except Exception as e:
                print(f"[Exploration] Picking '{obj}' failed:", e)

        # 3. Try every drawer/handle combination for exploration via "execute_pull" (lock knowledge)
        for drawer in drawers:
            # Find corresponding handle for this drawer if known
            handle = None
            for h in handles:
                if drawer in h:
                    handle = h
                    break
            # Move to the drawer's location
            drawer_info = positions.get(drawer, {})
            dr_loc = drawer_info.get('location', None) if isinstance(drawer_info, dict) else None
            if dr_loc and robot_loc != dr_loc:
                try:
                    print(f"[Exploration] Moving to '{dr_loc}' to try pulling '{drawer}'.")
                    obs, reward, done = execute_go(
                        env,
                        task,
                        start=robot_loc,
                        goal=dr_loc,
                        max_steps=100,
                        threshold=0.01
                    )
                    robot_loc = dr_loc
                except Exception as e:
                    print(f"[Exploration] Can't move to {dr_loc} for '{drawer}':", e)
                    continue
            # Try pulling (requires handle)? Or just the drawer?
            try:
                if handle:
                    print(f"[Exploration] Trying to pull '{drawer}' with handle '{handle}'.")
                    obs, reward, done = execute_pull(
                        env,
                        task,
                        drawer=drawer,
                        handle=handle,
                        target_pos=dr_loc,
                        max_steps=100,
                        threshold=0.01,
                        timeout=10.0
                    )
                else:
                    print(f"[Exploration] Trying to pull '{drawer}'.")
                    obs, reward, done = execute_pull(
                        env,
                        task,
                        drawer=drawer,
                        target_pos=dr_loc,
                        max_steps=100,
                        threshold=0.01,
                        timeout=10.0
                    )
                print(f"[Exploration] Pulled '{drawer}'.")
            except Exception as e:
                print(f"[Exploration] Pull on '{drawer}' failed:", e)

        # 4. Try "execute_sweep" on all objects at every location, in case it's an exploration primitive
        for obj in objects:
            obj_info = positions.get(obj, {})
            if isinstance(obj_info, dict) and 'location' in obj_info:
                obj_loc = obj_info['location']
            elif isinstance(obj_info, str):
                obj_loc = obj_info
            else:
                continue
            if robot_loc != obj_loc:
                try:
                    print(f"[Exploration] Moving to '{obj_loc}' to try sweeping '{obj}'.")
                    obs, reward, done = execute_go(
                        env,
                        task,
                        start=robot_loc,
                        goal=obj_loc,
                        max_steps=100,
                        threshold=0.01
                    )
                    robot_loc = obj_loc
                except Exception as e:
                    print(f"[Exploration] Can't get to {obj_loc} for sweep:", e)
                    continue
            try:
                print(f"[Exploration] Trying to sweep '{obj}' at '{obj_loc}'.")
                obs, reward, done = execute_sweep(
                    env,
                    task,
                    target_object=obj,
                    target_pos=obj_loc,
                    max_steps=100,
                    threshold=0.01
                )
            except Exception as e:
                print(f"[Exploration] Sweep for '{obj}' failed:", e)
        
        # Record that exploration has been attempted
        print("[Exploration] Exploration phase completed. If missing preconditions exist, knowledge predicates should now be acquired.")

        # === END OF EXPLORATION PHASE ===

        # # -----------------------------------------------------------------
        # # === MAIN TASK PLAN EXECUTION HERE (after exploration phase) ===
        # # (COMMENTED OUT; to be replaced/filled by oracle plan steps directly)
        # # Sample plan steps (uncomment and adapt as needed):
        #
        # plan = [
        #     ("execute_go", "locA", "locB"),
        #     ("execute_pick", "obj1", "locB"),
        #     ("execute_go", "locB", "locC"),
        #     ("execute_place", "obj1", "drawer1", "locC"),
        # ]
        # for step in plan:
        #     skill = step[0]
        #     args = step[1:]
        #     try:
        #         print(f"[Plan] Executing {skill} with args {args} ...")
        #         if skill == "execute_go":
        #             obs, reward, done = execute_go(
        #                 env,
        #                 task,
        #                 start=args[0],
        #                 goal=args[1],
        #                 max_steps=100,
        #                 threshold=0.01
        #             )
        #         elif skill == "execute_pick":
        #             obs, reward, done = execute_pick(
        #                 env,
        #                 task,
        #                 target_object=args[0],
        #                 target_pos=args[1],
        #                 max_steps=100,
        #                 approach_distance=0.15,
        #                 threshold=0.01,
        #                 approach_axis='z',
        #                 timeout=10.0
        #             )
        #         elif skill == "execute_place":
        #             obs, reward, done = execute_place(
        #                 env,
        #                 task,
        #                 object=args[0],
        #                 drawer=args[1],
        #                 target_pos=args[2],
        #                 max_steps=100,
        #                 threshold=0.01,
        #                 timeout=10.0
        #             )
        #         ...
        #         print(f"[Plan] Step complete.")
        #         if done:
        #             print("[Plan] Task finished early.")
        #             break
        #     except Exception as e:
        #         print("[Plan] Step failed due to error:", e)
        # -----------------------------------------------------------------

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()