# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only externally provided skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example: positions = {'drawer1': (x, y, z), 'handle1': (x, y, z), ...}
        try:
            positions = get_object_positions()
        except Exception as e:
            print("[Error] Failed to get object positions:", e)
            positions = {}

        # === EXPLORATION PHASE ===
        # The feedback indicates the missing predicate is (drawer-closed drawer1)
        # We must explore and verify if drawer1 is closed, or cause the detection of this predicate.
        # As per Exploration Knowledge, we can use execute_pull to discover lock-known (but not drawer-closed directly).
        # However, in the primitive_skills_static_lock_v2 domain, opening (pulling) a drawer requires 'drawer-closed' to be true.
        # Thus, let's attempt to use the primitives to gather drawer state information.

        # For exploration, try to use execute_go to drawer location, then try execute_pull on the drawer handle.
        # This may cause predicates or errors that reveal missing knowledge.

        # Identify relevant objects in the scene
        drawer_name = 'drawer1'
        drawer_handle = None
        for key in positions:
            if 'handle' in key and 'drawer1' in key:
                drawer_handle = key
                break
        if drawer_handle is None:
            # Fallback: pick any handle
            for key in positions:
                if 'handle' in key:
                    drawer_handle = key
                    break

        # Get locations (assume drawer and handle are co-located)
        try:
            drawer_pos = positions[drawer_name]
        except KeyError:
            drawer_pos = None
        if drawer_handle:
            handle_pos = positions[drawer_handle]
        else:
            handle_pos = None

        # Find a nearby location to simulate a 'location' (for compatibility with primitive skills)
        # Fallback: use origin
        location = (0, 0, 0)
        if drawer_pos is not None:
            location = drawer_pos

        # Try to ensure robot is at the drawer location before interacting
        # Use execute_go (move) if available
        robot_current_location = None
        for key in positions:
            if 'robot' in key:
                robot_current_location = positions[key]
                break

        print("[Exploration] Moving robot to drawer location for exploration.")
        try:
            # Using placeholder names; in real code, pass actual identifiers required by your skills
            obs, reward, done = execute_go(
                env,
                task,
                from_location='init',  # or use the robot's current location if known
                to_location=location,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
        except Exception as e:
            print("[Exploration] execute_go failed or not required:", e)

        # Try to pick up the handle (needed for execute_pull)
        if drawer_handle:
            print("[Exploration] Attempting to pick up handle at:", handle_pos)
            try:
                obs, reward, done = execute_pick(
                    env,
                    task,
                    target_name=drawer_handle,
                    target_pos=handle_pos,
                    approach_distance=0.10,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
            except Exception as e:
                print("[Exploration] execute_pick handle failed:", e)
        else:
            print("[Exploration] No handle object found for drawer1.")

        # Try to pull the drawer using the handle
        if drawer_handle:
            print("[Exploration] Attempting to pull drawer using handle:", drawer_handle)
            try:
                obs, reward, done = execute_pull(
                    env,
                    task,
                    drawer_name=drawer_name,
                    handle_name=drawer_handle,
                    location=location,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                print("[Exploration] execute_pull succeeded. Drawer might be open now.")
            except Exception as e:
                print("[Exploration] execute_pull failed (drawer might be locked or already open/closed):", e)
                print("[Exploration] The error or feedback may indicate that the 'drawer-closed' predicate is relevant.")
        else:
            print("[Exploration] Skipping execute_pull - handle not found.")

        # At this point, observation or feedback should clarify that (drawer-closed drawer1)
        # is required/missing, fulfilling the exploration requirement.

        # === END EXPLORATION PHASE ===

        # The core oracle plan logic would continue below – e.g., using execute_pick, execute_place, etc.
        # For demonstration, this code focuses on the exploration phase as required.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()