# run_skeleton_task.py (Filled - Exploration for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only pre-defined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # positions: dict mapping object names to their positions or status
        positions = get_object_positions()

        # === Exploration Phase: Find Missing Predicate ===
        #
        # According to the feedback, the planner times out—likely because a required predicate
        # for a skill or action's precondition is missing in the environment state.
        #
        # From the main domain PDDL and the exploration domain, we suspect predicates like:
        #   - "lock-known", "identified", "temperature-known", "weight-known", "durability-known"
        #   - possibly one of the drawer state predicates is not set.
        #
        # Here, we must perform skill invocations that use available actions, observing which predicate 
        # or knowledge is missing by running various skills and handling any exceptions.
        #
        # Only the following skills may be used (must match: ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'])

        # Exploration plan to determine missing predicate/state
        # Example: Try to (move robot), (pick up obj), (open drawer), (place obj in drawer), ...
        # Wrap each in try/except to observe failures.

        # Retrieve environment-specific objects and locations
        # Fill these automatically if possible, or try all available objects
        objects = [k for k in positions if "object" in k or "obj" in k or "handle" in k or "drawer" in k]
        drawers = [k for k in positions if "drawer" in k]
        handles = [k for k in positions if "handle" in k]
        locations = [k for k in positions if "loc" in k or "room" in k or "table" in k or "ground" in k]

        # For robustness, create dummy variables if not detected
        if not locations and positions:
            # Assume all positions are locations for brute-force
            locations = list(positions.keys())

        # Identify robot's initial location (by querying predicates or guessing)
        robot_location = None
        for loc in locations:
            if "robot" in loc or "init" in loc or "start" in loc:
                robot_location = loc
                break
        if not robot_location and locations:
            robot_location = locations[0]  # Guess starting point

        print("[Exploration] All objects:", objects)
        print("[Exploration] All drawers:", drawers)
        print("[Exploration] All handles:", handles)
        print("[Exploration] All locations:", locations)
        print("[Exploration] Robot initial location:", robot_location)
        print("[Exploration] Raw object positions:", positions)

        # -- Attempt move to all locations with execute_go, see if predicate errors arise
        for to_loc in locations:
            if to_loc == robot_location:
                continue
            print(f"[Exploration] Trying execute_go: {robot_location} -> {to_loc}")
            try:
                result = execute_go(env, robot_location, to_loc)
                print(f"   Success: execute_go from {robot_location} to {to_loc}")
                robot_location = to_loc
            except Exception as e:
                print(f"   Exception in execute_go: {str(e)}")
                # If missing predicate referenced, log for debugging
                continue

        # -- Try picking up every "object" that isn't a handle or a drawer
        for obj in objects:
            if "handle" in obj or "drawer" in obj:
                continue
            for loc in locations:
                print(f"[Exploration] Trying execute_pick: {obj} at {loc}")
                try:
                    result = execute_pick(env, obj, loc)
                    print(f"   Success: execute_pick {obj} at {loc}")
                except Exception as e:
                    print(f"   Exception in execute_pick: {str(e)}")
                    # Look for missing predicate info in error message
                    continue

        # -- Try opening each drawer using a handle ("execute_pull")
        for drawer in drawers:
            # Try all handles attached to this drawer, at all locations
            for handle in handles:
                for loc in locations:
                    print(f"[Exploration] Trying execute_pull: {drawer}, handle: {handle}, at {loc}")
                    try:
                        # Must pick the handle first (since precondition: holding ?h)
                        try:
                            execute_pick(env, handle, loc)
                        except Exception as pe:
                            print(f"     skip execute_pick-handle: {handle} at {loc}: {str(pe)}")
                        result = execute_pull(env, drawer, handle, loc)
                        print(f"   Success: execute_pull {drawer} with {handle} at {loc}")
                    except Exception as e:
                        print(f"   Exception in execute_pull: {str(e)}")
                        continue

        # -- Try placing every object into every drawer at every location
        for obj in objects:
            if "handle" in obj or "drawer" in obj:
                continue
            for drawer in drawers:
                for loc in locations:
                    print(f"[Exploration] Trying execute_place: {obj} in {drawer} at {loc}")
                    try:
                        # Must pick the object first
                        try:
                            execute_pick(env, obj, loc)
                        except Exception as pe:
                            print(f"     skip execute_pick-object: {obj} at {loc}: {str(pe)}")
                        result = execute_place(env, obj, drawer, loc)
                        print(f"   Success: execute_place {obj} in {drawer} at {loc}")
                    except Exception as e:
                        print(f"   Exception in execute_place: {str(e)}")
                        continue

        # -- Try pushing (closing) all drawers at all locations
        for drawer in drawers:
            for loc in locations:
                print(f"[Exploration] Trying execute_push: {drawer} at {loc}")
                try:
                    result = execute_push(env, drawer, loc)
                    print(f"   Success: execute_push {drawer} at {loc}")
                except Exception as e:
                    print(f"   Exception in execute_push: {str(e)}")
                    continue

        # -- Try other available skills, e.g., execute_sweep, execute_rotate, execute_gripper
        for obj in objects:
            for loc in locations:
                print(f"[Exploration] Trying execute_sweep: {obj} at {loc}")
                try:
                    result = execute_sweep(env, obj, loc)
                    print(f"   Success: execute_sweep {obj} at {loc}")
                except Exception as e:
                    print(f"   Exception in execute_sweep: {str(e)}")
                    continue

        print("[Exploration] Completed skill probing for missing predicates.")
        print("Check exception outputs for clues about missing required state/predicate.")
        # At this point, exceptions may mention missing predicates like "drawer-locked", "drawer-unlocked", etc.
        # which are required for further domain debugging.
        #
        # Now, based on the feedback/exception content, the user can update the initial state.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
