# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate Inference)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use all skills as provided

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, including exploration to determine missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Gets a dict mapping object names to positions
        positions = get_object_positions()

        # === Exploration Phase ===
        # Since the planner failed (timeout), we attempt to determine if the problem is caused by a missing predicate,
        # such as a known 'locked/unlocked' or 'identified' predicate.
        #
        # We attempt skill executions that could help the planner by building the knowledge predicates
        # (like "lock-known", "identified", etc.), using available exploration knowledge.
        #
        # This phase runs before the (presumed) oracle plan.

        print("[Exploration] Commencing predicate exploration phase...")

        # Example object and location inference from positions
        # These should be replaced by actual object/location names from the environment if needed
        all_objects = [name for name in positions if not name.startswith('loc_') and not name.startswith('drawer_')]
        all_locations = [name for name in positions if name.startswith('loc_') or name.startswith('room')]
        all_drawers = [name for name in positions if name.startswith('drawer_')]
        robot_location = None
        if "robot" in positions:
            robot_location = positions["robot"]

        # For this exploration, just pick the first robot location
        if len(all_locations) == 0:
            print("[Exploration] No locations found in environment!")
        else:
            # We'll use the first location, but in a real code base, we might want to query current robot pose
            robot_current_location = all_locations[0]
        
        # Try to identify which exploration action might be missing by stepping through plausible exploration skills
        # Use only predefined skills from available skills list

        exploration_needed = True

        # We need to call the relevant skills for "identifying", "temperature-known", etc.,
        # but only those skills present in our available skill set
        available_skills = ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']
        
        # If the domain is missing some predicate (e.g. "lock-known", "identified"), planner may require them.
        # So try available skills to induce these predicates.

        # Example: Try pulling on drawer handles to ensure lock-known predicate is set
        for drawer in all_drawers:
            handle_name = None
            # Try to find a handle associated with this drawer
            # Try common handle naming convention
            candidate_handles = [obj for obj in all_objects if "handle" in obj and drawer in obj]
            if candidate_handles:
                handle_name = candidate_handles[0]
            else:
                candidate_handles = [obj for obj in all_objects if "handle" in obj]
                if candidate_handles:
                    handle_name = candidate_handles[0]
            if handle_name is None:
                continue

            for robot_loc in all_locations:
                try:
                    # === Step 1: Go to location of drawer ===
                    print(f"[Exploration] Moving robot to location {robot_loc} for drawer {drawer}...")
                    if 'execute_go' in available_skills:
                        obs, reward, done = execute_go(env, task, from_location=robot_current_location, to_location=robot_loc)
                        robot_current_location = robot_loc
                        if done:
                            print("[Exploration] Task ended while exploring (move)!")
                            return

                    # === Step 2: Pick handle ===
                    print(f"[Exploration] Picking handle {handle_name} at {robot_loc}...")
                    if 'execute_pick' in available_skills:
                        obs, reward, done = execute_pick(env, task, obj=handle_name, location=robot_loc)
                        if done:
                            print("[Exploration] Task ended while exploring (pick handle)!")
                            return

                    # === Step 3: Pull on handle (check for lock-known) ===
                    print(f"[Exploration] Pulling on handle {handle_name} (drawer {drawer}) at {robot_loc}...")
                    if 'execute_pull' in available_skills:
                        obs, reward, done = execute_pull(env, task, drawer=drawer, h=handle_name, p=robot_loc)
                        if done:
                            print("[Exploration] Task ended while exploring (pull)!")
                            return

                except Exception as e:
                    print(f"[Exploration] Exception during exploration on drawer '{drawer}': {e}")
                    continue

        # Try identifying objects at their locations (if such skill existed)
        # We'll illustrate with a sweep, to ensure object is noticed on the floor
        for obj in all_objects:
            for loc in all_locations:
                try:
                    print(f"[Exploration] Sweeping object {obj} at location {loc}...")
                    if 'execute_sweep' in available_skills:
                        obs, reward, done = execute_sweep(env, task, o=obj, p=loc)
                        if done:
                            print("[Exploration] Task ended while exploring (sweep)!")
                            return
                except Exception as e:
                    print(f"[Exploration] Exception during sweep exploration for object '{obj}': {e}")
                    continue

        print("[Exploration] Exploration phase complete. Proceeding to task plan (oracle plan if provided).")

        # === Oracle/Task Plan Execution Phase ===
        # TODO: The plan should be derived from the specific goal/problem.
        # Here, we'd normally execute the oracle plan steps,
        # mapping each to a skill invocation using only skills available in skill_code.

        # -- Example placeholder for plan execution logic --
        # For each plan step: determine action name and arguments, then call skill function
        # You would replace the following with actual plan steps, e.g.:
        # obs, reward, done = execute_pick(env, task, obj='obj1', location='room1')
        # obs, reward, done = execute_place(env, task, obj='obj1', drawer='drawer1', p='room1')
        #
        # If 'done' is True after any step, exit the routine.
        #
        # (No generic plan provided, so only exploration logic is included.)

        print("===== Task completed (no explicit plan to execute after exploration) =====")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
