# run_skeleton_task.py (Completed with Exploration and Predicate Checking)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # All skills are imported, do not redefine them

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers (optional)
        init_video_writers(obs)

        # Wrap the task steps for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        print("[Debug] Retrieved object positions: ", positions)
        # Example: positions = {'drawer': (x, y, z), 'apple': (x, y, z), ...}

        # === Start Exploration for Missing Predicate ===
        print("===== Exploration Phase =====")

        # List available locations and objects for exploration
        locations = [k for k,v in positions.items() if 'location' in k or 'room' in k or 'table' in k or 'drawer' in k]
        objects = [k for k in positions.keys() if k not in locations]

        # Suppose the robot starts at its default position in positions; find the starting location
        robot_location = None
        for name in positions:
            if 'robot' in name or 'start' in name:
                robot_location = name
                break

        # Fallback for robot_location if not explicitly present
        if robot_location is None and locations:
            robot_location = locations[0]

        # 1. Explore to identify objects at locations
        for loc in locations:
            try:
                if robot_location != loc:
                    print(f"[Exploration] execute_go: {robot_location} -> {loc}")
                    obs, reward, done = execute_go(env, task, robot_location, loc)
                    robot_location = loc
                print(f"[Exploration] Locating/identifying objects at location: {loc}")
                # Use available skills -- substitute for exploration, e.g., we may need to 'execute_sweep' to sense,
                # or just print objects at this position.
                # Here, call execute_sweep (if suitable) to 'explore' this location:
                for obj in objects:
                    try:
                        print(f"[Exploration] execute_sweep on object: {obj} at {loc}")
                        obs, reward, done = execute_sweep(env, task, obj, loc)
                    except Exception as e:
                        print(f"[Warning] Could not sweep object {obj} at {loc}: {str(e)}")
            except Exception as e:
                print(f"[Warning] Could not move to {loc}: {str(e)}")

        # If strong exploration/skill required, could try pick/place/pull actions, but don't invent new skills

        print("===== Exploration Complete =====")
        print("-> Based on feedback, check for missing predicate in preconditions/effects using skill and observation.")

        # Normally, would check what predicates are missing in the initial state (as per plan failure).
        # Here, for demonstration, we assume 'drawer-closed' is sometimes missing for actions like execute_pull.
        #
        # Example: If a drawer exists in positions, try to pull/open it, but first check its status.

        # Example sequence: Go to drawer, try to open it
        for obj in objects:
            if 'drawer' in obj:
                drawer_name = obj
                handle_name = None
                # Try to find a matching handle for this drawer
                for o in objects:
                    if 'handle' in o and o in positions:
                        handle_name = o
                        break
                # Move to drawer location (if not already there)
                drawer_loc = None
                if drawer_name in locations:
                    drawer_loc = drawer_name
                elif any(l in drawer_name for l in locations):
                    drawer_loc = [l for l in locations if l in drawer_name][0]
                elif locations:
                    drawer_loc = locations[0]
                if robot_location != drawer_loc and drawer_loc is not None:
                    print(f"[Action] execute_go: {robot_location} -> {drawer_loc}")
                    try:
                        obs, reward, done = execute_go(env, task, robot_location, drawer_loc)
                        robot_location = drawer_loc
                    except Exception as e:
                        print(f"[Warning] Could not move to {drawer_loc}: {str(e)}")
                # Try to pick the handle (if it exists) at this drawer
                if handle_name is not None:
                    try:
                        print(f"[Action] execute_pick: {handle_name} at {drawer_loc}")
                        obs, reward, done = execute_pick(env, task, handle_name, drawer_loc)
                    except Exception as e:
                        print(f"[Warning] Could not pick handle {handle_name} at {drawer_loc}: {str(e)}")
                # Try to pull (open) the drawer
                try:
                    print(f"[Action] execute_pull: {drawer_name}, {handle_name}, {drawer_loc}")
                    obs, reward, done = execute_pull(env, task, drawer_name, handle_name, drawer_loc)
                    print(f"[Action] execute_pull result: done={done}")
                except Exception as e:
                    print(f"[Warning] Could not pull drawer {drawer_name}: {str(e)}")
                # Optionally, place an object in the drawer
                for o in objects:
                    if o != drawer_name and o != handle_name:
                        try:
                            print(f"[Action] execute_pick: {o} at {drawer_loc}")
                            obs, reward, done = execute_pick(env, task, o, drawer_loc)
                        except Exception as e:
                            print(f"[Warning] Could not pick object {o}: {str(e)}")
                        try:
                            print(f"[Action] execute_place: {o} in {drawer_name} at {drawer_loc}")
                            obs, reward, done = execute_place(env, task, o, drawer_name, drawer_loc)
                        except Exception as e:
                            print(f"[Warning] Could not place object {o} in {drawer_name}: {str(e)}")
                break  # Operate on only one drawer for demo

        # If no drawer exists, demonstrate pick-and-place on any two objects/locations
        if not any('drawer' in obj for obj in objects):
            # Pick the first object and move it to the first location
            if objects and locations:
                src_obj = objects[0]
                dst_loc = locations[0]
                try:
                    print(f"[Fallback] execute_pick: {src_obj} at {dst_loc}")
                    obs, reward, done = execute_pick(env, task, src_obj, dst_loc)
                except Exception as e:
                    print(f"[Warning] Could not pick object {src_obj} at {dst_loc}: {str(e)}")
                try:
                    print(f"[Fallback] execute_go: {dst_loc} -> {locations[-1]}")
                    obs, reward, done = execute_go(env, task, dst_loc, locations[-1])
                except Exception as e:
                    print(f"[Warning] Could not move: {str(e)}")
                try:
                    print(f"[Fallback] execute_place: {src_obj} at {locations[-1]}")
                    obs, reward, done = execute_place(env, task, src_obj, None, locations[-1])
                except Exception as e:
                    print(f"[Warning] Could not place object {src_obj} at {locations[-1]}: {str(e)}")

        print("===== Skeleton Task Logic Complete =====")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
