# run_skeleton_task.py (Completed for Exploration of Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use externally defined skills directly

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton: augmented for exploration (missing predicate detection)'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (0,0,0), ...}
        positions = get_object_positions()

        # ----------------------------------------------------------------------
        # Exploration Phase: Determine/Diagnose Missing Predicate using available skills
        # ----------------------------------------------------------------------

        # ---- Step 1: Check if there is a handle_pose (or similar) location ----
        # The feedback suggests robot-at handle_pose is not available/predicate missing.
        if 'handle_pose' in positions:
            handle_pose = positions['handle_pose']
            print(f"[Exploration] Detected handle_pose at {handle_pose}")
        else:
            print("[Exploration] handle_pose location not found in positions!")
            # Can't proceed with test. Continue to other exploration or raise exception.
        
        # Directly retrieve starting location; adapt to your setup.
        # Here, try to find the robot's starting location:
        robot_start_location = None
        for key in positions:
            if key.startswith('robot') or key == 'robot_start':
                robot_start_location = positions[key]
                break
        if robot_start_location is None:
            # Fallback: try some default or raise error
            robot_start_location = (0,0,0)
            print("[Exploration] Robot start location not found, using default (0,0,0).")
        
        # ---- Step 2: Use execute_go or execute_go_identify to reach handle_pose ----
        # We try to move the robot to handle_pose using available skills.
        # In our skills list, only execute_go is defined in the primitive domain, so we use it.
        print("[Exploration] Attempting to move robot to handle_pose via execute_go.")
        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_pos=robot_start_location,
                to_pos=handle_pose,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print("[Exploration] Robot moved to handle_pose.")
        except Exception as e:
            print(f"[Exploration][Error] Failed to move robot to handle_pose: {e}")

        # ---- Step 3: Try to check which PDDL predicates have changed/are available ----
        # For diagnosis, try to perform a skill that should check/use robot-at (handle_pose).
        # Since feedback says (robot-at handle_pose) is missing, we output after movement.
        print("[Exploration] Checking for predicate 'robot-at handle_pose'. (Observation required)")

        # Optionally, print current robot position as seen by environment for manual check:
        current_positions = get_object_positions()
        print(f"[Exploration] Current object/robot positions after move: {current_positions}")

        # ---- Step 4: Try to execute other investigation skill if possible ----
        # If our domain included 'execute_go_identify', use it (here only execute_go is provided).
        # Print note to signal what the plan would be if predicate was present.
        print("[Exploration] If execute_go_identify existed, it would record (identified ...) at handle_pose.")
        print("[Exploration] Based on feedback, (robot-at handle_pose) may be missing from PDDL state.")

        # ---- Step 5: End exploration, shutdown environment ----

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()