# run_skeleton_task.py (Filled for Exploration of Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only these predefined skill functions

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing the simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object and Drawer/Location Information ===
        positions = get_object_positions()

        # -- Exploration PHASE: Try to discover the missing predicate or state through action execution --

        # In the given context, the feedback is:
        # (drawer-open drawer_bottom)
        # This suggests the ability to recognize whether a drawer (drawer_bottom) is open
        # To find the missing predicate, we must use only available skills and PDDL knowledge

        # The action most related to drawer state is: execute_pull, execute_push, execute_go, execute_pick, execute_place
        # The exploration domain shows that "execute_pull" in exploration checks lock-known, but in the main domain, execute_pull opens the drawer
        # Our aim is to "explore" what predicate is missing to achieve (drawer-open drawer_bottom)
        # So, try to open the drawer and check which predicates change in the environment
        
        # Identify relevant objects (drawers, handles, locations)
        drawers = []
        handles = []
        locations = []
        for obj_name, info in positions.items():
            if 'drawer' in obj_name:
                drawers.append(obj_name)
            if 'handle' in obj_name:
                handles.append(obj_name)
            if info.get('type', '') == 'location':
                locations.append(obj_name)
        
        # Make some safe fallbacks if none found
        if len(drawers) == 0:
            drawers = ['drawer_bottom']
        if len(handles) == 0:
            handles = ['handle_bottom']
        if len(locations) == 0:
            locations = ['drawer_area']  # generic

        # Choose target entities for exploration
        drawer = drawers[0]
        handle = handles[0]
        robot_location = locations[0] if locations else None

        # === Step 1: Move robot to location of the drawer
        try:
            if robot_location:
                # Get current robot position if provided by obs or task
                robot_at = None
                try:
                    robot_at = obs['robot_state']['position_name']  # adapt as needed
                except Exception:
                    pass
                if not robot_at:
                    robot_at = robot_location
                execute_go(env, task, from_location=robot_at, to_location=robot_location)
                print(f"[Exploration] Robot moved to location: {robot_location}")
        except Exception as e:
            print("[Exploration] Error during robot move:", e)

        # === Step 2: Try to pick the handle (requires hand to be empty, handle must be on floor, etc.)
        try:
            execute_pick(env, task, obj=handle, location=robot_location)
            print(f"[Exploration] Picked up handle: {handle}")
        except Exception as e:
            print("[Exploration] Error during handle pick:", e)

        # === Step 3: Try to open the drawer using the handle
        try:
            execute_pull(env, task, drawer=drawer, handle=handle, location=robot_location)
            print(f"[Exploration] Attempted to open drawer: {drawer} using handle: {handle}")
        except Exception as e:
            print("[Exploration] Error during drawer open (pull):", e)
        
        # === Step 4: After the pull action, check which predicates are now true (detect missing predicate)
        # We try to read the environment state and print any changes related to drawer's state
        state_predicates = {}
        try:
            post_obs = task.get_observation()
            state_predicates = post_obs.get('predicates', {})
            # Print relevant predicates for all drawers
            print("[Exploration] Drawer state after pull:")
            for d in drawers:
                predicates = []
                for pred in ['drawer-open', 'drawer-closed', 'drawer-locked', 'drawer-unlocked', 
                             'drawer-full', 'drawer-empty', 'drawer-partially-filled']:
                    pred_str = f"{pred} {d}"
                    if pred_str in state_predicates:
                        predicates.append(pred_str)
                print(f"  Drawer {d}: {predicates}")
        except Exception as e:
            print("[Exploration] Error during state predicate inspection:", e)
            print("[Exploration] You may need to check the state manually or via visual output.")

        # --- End of Exploration Phase ---
        print("===== Finished Exploration Phase =====")

        # (Optional) If the exploration phase reveals the missing predicate (e.g., 'drawer-open'), you can proceed to use this information in later planning

        # === (End) ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
