# run_skeleton_task.py (Executable Exploration for Missing Predicate Detection)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use all provided primitive skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for predicate exploration using only available/existing skills.'''
    print("===== Starting Skeleton Task (Exploration) =====")
    
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        positions = get_object_positions()

        # === Exploration phase for missing predicate ===
        # This phase tests available skills and observes feedback/errors to discover which predicate is missing

        # Candidate skills inferred from Exploration Knowledge/domain PDDL:
        candidate_skills = [
            'execute_pick',
            'execute_place',
            'execute_push',
            'execute_pull',
            'execute_sweep',
            'execute_rotate',
            'execute_go',
            'execute_gripper'
        ]

        # Try to find at least one obtainable (object, location, drawer) tuple
        # (In practice, you may want to get this from the observation and PDDL objects, but here we assume object_positions returns something usable)

        # Extract likely entities for the primitives
        # Assumptions for demonstration (to be replaced with real env data in live environment):
        objects = [k for k in positions.keys() if "drawer" not in k and "handle" not in k and "location" not in k]
        drawers = [k for k in positions.keys() if "drawer" in k]
        handles = [k for k in positions.keys() if "handle" in k]
        locations = [k for k in positions.keys() if "location" in k]
        if len(locations) == 0 and 'robot' in positions:
            # Sometimes the robot's position is named, or we designate a default
            locations = ['default_location']

        # Explore all basic skills systematically to find which action fails with a predicate-related error
        exploration_log = []
        for obj_name in objects:
            # Try picking up the object at each location
            for loc in locations:
                try:
                    print(f"Exploring: Attempting execute_pick for object '{obj_name}' at location '{loc}'")
                    obs, reward, done = execute_pick(env, task, obj_name, loc)
                    exploration_log.append(('execute_pick', obj_name, loc, 'success'))
                except Exception as e:
                    print(f"[Exploration] execute_pick({obj_name}, {loc}) failed: {e}")
                    exploration_log.append(('execute_pick', obj_name, loc, f'fail: {e}'))
                    # If the error message mentions a missing predicate, log and break
                    if "predicate" in str(e).lower():
                        print("Likely missing predicate detected during 'execute_pick'")
                        break

        for dr_name in drawers:
            for loc in locations:
                try:
                    print(f"Exploring: Attempting execute_push for drawer '{dr_name}' at location '{loc}'")
                    obs, reward, done = execute_push(env, task, dr_name, loc)
                    exploration_log.append(('execute_push', dr_name, loc, 'success'))
                except Exception as e:
                    print(f"[Exploration] execute_push({dr_name}, {loc}) failed: {e}")
                    exploration_log.append(('execute_push', dr_name, loc, f'fail: {e}'))
                    
                try:
                    print(f"Exploring: Attempting execute_pull for drawer '{dr_name}' using a handle, at location '{loc}'")
                    # We'll use any handle we have—if none, skip
                    h_name = handles[0] if handles else None
                    if h_name:
                        obs, reward, done = execute_pull(env, task, dr_name, h_name, loc)
                        exploration_log.append(('execute_pull', dr_name, h_name, loc, 'success'))
                except Exception as e:
                    print(f"[Exploration] execute_pull({dr_name}, handle, {loc}) failed: {e}")
                    exploration_log.append(('execute_pull', dr_name, h_name if handles else 'handle_none', loc, f'fail: {e}'))

        # Try navigation action
        for loc_from in locations:
            for loc_to in locations:
                if loc_from != loc_to:
                    try:
                        print(f"Exploring: Attempting execute_go from '{loc_from}' to '{loc_to}'")
                        obs, reward, done = execute_go(env, task, loc_from, loc_to)
                        exploration_log.append(('execute_go', loc_from, loc_to, 'success'))
                    except Exception as e:
                        print(f"[Exploration] execute_go({loc_from}, {loc_to}) failed: {e}")
                        exploration_log.append(('execute_go', loc_from, loc_to, f'fail: {e}'))
        
        # Test the rest of the primitives with generic parameters:
        for obj_name in objects:
            for loc in locations:
                try:
                    print(f"Exploring: Attempting execute_sweep for object '{obj_name}' at '{loc}'")
                    obs, reward, done = execute_sweep(env, task, obj_name, loc)
                    exploration_log.append(('execute_sweep', obj_name, loc, 'success'))
                except Exception as e:
                    print(f"[Exploration] execute_sweep({obj_name}, {loc}) failed: {e}")
                    exploration_log.append(('execute_sweep', obj_name, loc, f'fail: {e}'))
                try:
                    print(f"Exploring: Attempting execute_rotate for object '{obj_name}' at '{loc}'")
                    obs, reward, done = execute_rotate(env, task, obj_name, loc)
                    exploration_log.append(('execute_rotate', obj_name, loc, 'success'))
                except Exception as e:
                    print(f"[Exploration] execute_rotate({obj_name}, {loc}) failed: {e}")
                    exploration_log.append(('execute_rotate', obj_name, loc, f'fail: {e}'))

        # Test gripper skill
        try:
            print("Exploring: Attempting execute_gripper()")
            obs, reward, done = execute_gripper(env, task)
            exploration_log.append(('execute_gripper', 'success'))
        except Exception as e:
            print(f"[Exploration] execute_gripper() failed: {e}")
            exploration_log.append(('execute_gripper', f'fail: {e}'))

        print("===== Exploration results =====")
        for log_entry in exploration_log:
            print(log_entry)
        print("===== End of predicate exploration phase. Review errors and logs to find predicate issues. =====")

    finally:
        # Always ensure proper environment shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
