# run_skeleton_task.py (Exploration & Plan Execution Skeleton)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # All skills are imported directly

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Run exploration to find missing predicates and execute plan using provided skills.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)
        positions = get_object_positions()

        # -- Exploration Phase --
        # Purpose: Try each available skill in plausible combinations to deduce if an action fails,
        # likely due to a missing precondition/predicate in the domain or problem.
        skills = [
            "execute_pick",
            "execute_place",
            "execute_push",
            "execute_pull",
            "execute_sweep",
            "execute_rotate",
            "execute_go",
            "execute_gripper"
        ]

        # For this demonstration, we will only attempt parameter setups for
        # the first exploration of each skill — further refinement would iterate or learn from failures.
        exploration_results = {}

        # Try to infer plausible arguments from observation and object positions
        object_names = list(positions.keys())
        location_names = set()
        for obj in object_names:
            pos = positions[obj]
            if isinstance(pos, dict) and 'location' in pos:
                location_names.add(pos['location'])
        location_names = list(location_names)

        # Best effort guess: if we have some known types (object, location, drawer, etc.)
        # For exploration, make a best guess for types
        # Since parsing of the full :objects/:init isn't shown, we use available positions and skill likely arguments.

        # Find some objects for generic object parameters
        test_object = None
        for obj in object_names:
            if "handle" not in obj.lower() and "drawer" not in obj.lower():
                test_object = obj
                break
        # Find a drawer
        test_drawer = None
        for obj in object_names:
            if "drawer" in obj.lower():
                test_drawer = obj
                break
        # Find a handle
        test_handle = None
        for obj in object_names:
            if "handle" in obj.lower():
                test_handle = obj
                break

        # Try to obtain at least two locations
        if len(location_names) < 2:
            # fallback: make up two
            location_names = ['location_1', 'location_2']
        from_loc = location_names[0]
        to_loc = location_names[-1]

        # Run each skill with guessed parameters and store the result (whether error occurs)
        for skill in skills:
            try:
                if skill == "execute_pick" and test_object and from_loc:
                    print(f"[Exploration] Trying {skill}({test_object}, {from_loc})")
                    obs, reward, done = execute_pick(env, task, test_object, from_loc)
                    exploration_results[skill] = "success"
                elif skill == "execute_place" and test_object and test_drawer and from_loc:
                    print(f"[Exploration] Trying {skill}({test_object}, {test_drawer}, {from_loc})")
                    obs, reward, done = execute_place(env, task, test_object, test_drawer, from_loc)
                    exploration_results[skill] = "success"
                elif skill == "execute_push" and test_drawer and from_loc:
                    print(f"[Exploration] Trying {skill}({test_drawer}, {from_loc})")
                    obs, reward, done = execute_push(env, task, test_drawer, from_loc)
                    exploration_results[skill] = "success"
                elif skill == "execute_pull" and test_drawer and test_handle and from_loc:
                    print(f"[Exploration] Trying {skill}({test_drawer}, {test_handle}, {from_loc})")
                    obs, reward, done = execute_pull(env, task, test_drawer, test_handle, from_loc)
                    exploration_results[skill] = "success"
                elif skill == "execute_sweep" and test_object and from_loc:
                    print(f"[Exploration] Trying {skill}({test_object}, {from_loc})")
                    obs, reward, done = execute_sweep(env, task, test_object, from_loc)
                    exploration_results[skill] = "success"
                elif skill == "execute_rotate":
                    print(f"[Exploration] Trying {skill}()")
                    obs, reward, done = execute_rotate(env, task)
                    exploration_results[skill] = "success"
                elif skill == "execute_go" and from_loc and to_loc:
                    print(f"[Exploration] Trying {skill}({from_loc}, {to_loc})")
                    obs, reward, done = execute_go(env, task, from_loc, to_loc)
                    exploration_results[skill] = "success"
                elif skill == "execute_gripper":
                    print(f"[Exploration] Trying {skill}()")
                    obs, reward, done = execute_gripper(env, task)
                    exploration_results[skill] = "success"
                else:
                    exploration_results[skill] = "skipped (missing guessable arguments)"
            except Exception as e:
                print(f"[Exploration] {skill} failed with error:", e)
                exploration_results[skill] = f"failed: {e}"

        # Print result for developer info:
        print("\n===== Exploration Results (for predicate diagnosis) =====")
        for skill, res in exploration_results.items():
            print(f"  {skill}: {res}")

        # Developer should inspect which skill fails in a predictable way; possibly due to a missing predicate
        # In robust code, would adjust plan or suggest which predicate is absent/misconfigured

        # -- Task Plan Execution Placeholder --
        # At this point, proceed to run the oracle plan (not given here) if available
        # Replace with logic or calls as appropriate to your scenario and plan
        # For demonstration, we just log that the exploration (predicate checking) finished

    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
