# run_skeleton_task.py (Completed Executable)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Do not redefine primitives or skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with integrated exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Feedback and exploration context
        # We need to explore which predicate is missing for robot-at drawer-area

        # Safe defaults for required keys, update as per actual environment
        robot_current_location = positions.get('robot', {}).get('location', None)
        drawer_area = positions.get('drawer-area', None)
        if isinstance(drawer_area, dict):
            drawer_area_pos = drawer_area.get('position')
        else:
            drawer_area_pos = drawer_area
        # Find all drawers and handles (by keys or types, example below)
        drawer_name = None
        handle_name = None
        for obj in positions:
            if 'drawer' in obj and obj != 'drawer-area':
                drawer_name = obj
            if 'handle' in obj or 'grip' in obj:
                handle_name = obj

        # Fallback for object names if not found
        if not drawer_name:
            drawer_name = 'drawer'
        if not handle_name:
            handle_name = 'handle'
        # Find at least one object that can be picked from the floor
        object_on_floor = None
        for obj in positions:
            if obj.startswith("object") or obj.startswith("item"):
                object_on_floor = obj
                break
        if not object_on_floor:
            object_on_floor = 'object1'

        # == 1. Exploration Phase: Try to move the robot and check the effect ==
        # We want to check if after calling execute_go, the predicate (robot-at drawer-area) holds
        print("[Exploration] Moving robot to drawer-area using 'execute_go'.")
        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_current_location,
                to_location='drawer-area',  # use symbolic name as required by skill_code
                max_steps=100
            )
            # Update current robot location for further actions
            robot_current_location = 'drawer-area'
        except Exception as e:
            print(f"[ERROR] execute_go failed: {e}")
            obs, reward, done = None, 0.0, False

        # Post-move: Verify via feedback if (robot-at drawer-area) holds
        # We assume that the environment creates this predicate.
        # In a real scenario, here you could query state or check obs if possible

        # == 2. Task Execution Phase: Use available skills to interact with drawer ==
        #
        # Typically, the goal is: Pick up an object, move to drawer, open it, and place object inside.
        # The following is a generic flow based on available skills and domain:

        try:
            # --- Pick up object from floor ---
            print(f"[Task] Picking up '{object_on_floor}' from location '{robot_current_location}'.")
            obs, reward, done = execute_pick(
                env,
                task,
                obj_name=object_on_floor,
                location=robot_current_location,
                max_steps=100
            )

            # --- Open the drawer (with a handle) ---
            print(f"[Task] Picking up handle '{handle_name}' to pull open drawer '{drawer_name}'.")
            obs, reward, done = execute_pick(
                env,
                task,
                obj_name=handle_name,
                location=robot_current_location,
                max_steps=100
            )
            print(f"[Task] Pulling to open drawer '{drawer_name}'.")
            obs, reward, done = execute_pull(
                env,
                task,
                drawer_name=drawer_name,
                handle_name=handle_name,
                location=robot_current_location,
                max_steps=100
            )

            # --- Place the object in the drawer ---
            print(f"[Task] Placing '{object_on_floor}' into drawer '{drawer_name}' at '{robot_current_location}'.")
            obs, reward, done = execute_place(
                env,
                task,
                obj_name=object_on_floor,
                drawer_name=drawer_name,
                location=robot_current_location,
                max_steps=100
            )

            # --- Close the drawer ---
            print(f"[Task] Closing drawer '{drawer_name}' at '{robot_current_location}'.")
            obs, reward, done = execute_push(
                env,
                task,
                drawer_name=drawer_name,
                location=robot_current_location,
                max_steps=100
            )

            print("[Task] Task completed successfully!")

        except Exception as e:
            print(f"[ERROR] Exception occurred during task execution: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()