# run_skeleton_task.py (Completed to include exploration for missing predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Assume all primitives like execute_pick, execute_place, etc.

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, now with missing predicate exploration phase.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Find Missing Predicate ===
        # The feedback suggests the plan is not progressing because of a possible missing predicate (e.g. detection/knowledge precondition issue for locked/unlocked drawers or object properties).
        # We'll perform an exploration pass over all known skill actions to check which predicate is missing, by trialing each action and catching fails.
        print("[Exploration] Checking which predicates/actions lead to failures (e.g. due to missing knowledge)")

        # List of skills available
        skill_names = ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # Mapping from available skill to example parameters.
        # We'll construct dummy parameter sets using the positions and object info.
        # You may need to tune these according to your environment specifics!
        #
        # For exploration, we'll just try to call each skill with some plausible parameters to see which one fails due to a missing predicate.
        # In real PDDL, info like 'lock-known' may be required before an action can succeed (e.g., opening a drawer if lock state is not known).
        
        # Gather all objects, drawers, locations
        object_names = [k for k in positions.keys() if 'obj' in k or 'object' in k or 'handle' in k]
        drawer_names = [k for k in positions.keys() if 'drawer' in k]
        location_names = [k for k in positions.keys() if 'loc' in k or 'room' in k or 'place' in k]
        # Fallback to all position keys as locations if explicit locations not found
        if not location_names:
            location_names = list(positions.keys())

        exploration_results = {}
        for skill in skill_names:
            try:
                print(f"[Exploration] Trying skill: {skill}")
                # Choose plausible parameters
                if skill == 'execute_pick':
                    # Try to pick first object on first location
                    if object_names and location_names:
                        obs, reward, done = execute_pick(env, task, object_names[0], location_names[0])
                        exploration_results[skill] = (True, "")
                    else:
                        exploration_results[skill] = (False, "Missing object or location")
                elif skill == 'execute_place':
                    # Try placing first object in first drawer at first location
                    if object_names and drawer_names and location_names:
                        obs, reward, done = execute_place(env, task, object_names[0], drawer_names[0], location_names[0])
                        exploration_results[skill] = (True, "")
                    else:
                        exploration_results[skill] = (False, "Missing object, drawer, or location")
                elif skill == 'execute_push':
                    # Try pushing first drawer at first location
                    if drawer_names and location_names:
                        obs, reward, done = execute_push(env, task, drawer_names[0], location_names[0])
                        exploration_results[skill] = (True, "")
                    else:
                        exploration_results[skill] = (False, "Missing drawer or location")
                elif skill == 'execute_pull':
                    # Try pulling with first drawer, its handle, first location
                    handles = [k for k in object_names if 'handle' in k]
                    if drawer_names and handles and location_names:
                        obs, reward, done = execute_pull(env, task, drawer_names[0], handles[0], location_names[0])
                        exploration_results[skill] = (True, "")
                    else:
                        exploration_results[skill] = (False, "Missing drawer, handle, or location")
                elif skill == 'execute_sweep':
                    # Sweep first object at first location
                    if object_names and location_names:
                        obs, reward, done = execute_sweep(env, task, object_names[0], location_names[0])
                        exploration_results[skill] = (True, "")
                    else:
                        exploration_results[skill] = (False, "Missing object or location")
                elif skill == 'execute_rotate':
                    # Rotate first object at first location (if rotate takes those params)
                    if object_names and location_names:
                        obs, reward, done = execute_rotate(env, task, object_names[0], location_names[0])
                        exploration_results[skill] = (True, "")
                    else:
                        exploration_results[skill] = (False, "Missing object or location")
                elif skill == 'execute_go':
                    # Go from one location to another
                    if len(location_names) > 1:
                        obs, reward, done = execute_go(env, task, location_names[0], location_names[1])
                        exploration_results[skill] = (True, "")
                    else:
                        exploration_results[skill] = (False, "Not enough locations")
                elif skill == 'execute_gripper':
                    # Just call with task/env
                    obs, reward, done = execute_gripper(env, task)
                    exploration_results[skill] = (True, "")
                else:
                    exploration_results[skill] = (False, "Unknown skill")
            except Exception as e:
                exploration_results[skill] = (False, str(e))
                print(f"[Exploration] {skill} failed with error: {e}")

        print("[Exploration Results]")
        for skill, (ok, msg) in exploration_results.items():
            print(f"  Skill {skill}: {'Success' if ok else 'Fail'} {msg if msg else ''}")

        # --- End of exploration phase ---

        # If any action failed with an error message about predicate or knowledge, assume that's the missing predicate needed to progress.
        # (For example, 'lock-known' for drawers, or 'identified' for objects.)
        # In a real scenario, at this point you'd add a call to an exploration skill to establish that knowledge, e.g. go-identify or go-temperature etc.
        # For now, just report and proceed.

        print("[Task] Exploration complete. You can now construct your plan with the confirmed skills and required predicates.")
        
        # === (Task Plan Phase - not implemented here; would require oracle plan and goal description) ===
        # User should insert real task plan logic here using skill functions, now aware of which predicates/knowledge need to be established.
        # For example:
        #
        # print("[Task Plan] Moving robot to location to check properties.")
        # obs, reward, done = execute_go(env, task, current_location, target_location)
        # # ...etc...
        
        # --- Placeholder for actual plan ---
        print("[Task] You may safely insert your task plan here using the confirmed list of skills.")

    finally:
        # Ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
