# run_skeleton_task.py (Completed for Exploration & Missing Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use as per instructions

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Positions dictionary: object_name -> position tuple
        positions = get_object_positions()

        # ------------------ Exploration Phase ------------------
        # The planner timed out, suggesting an unknown/missing predicate or insufficient problem description.
        # We now run an exploration phase to observe the effects/preconditions of actions in the environment,
        # especially related to the known possible skills and the PDDL exploration domain from instructions.

        print("[Exploration] Starting predicate discovery and environment probing.")
        # First, build simple lists of objects and locations present
        all_objects = [obj for obj in positions.keys() if "drawer" not in obj and "handle" not in obj]
        all_drawers = [obj for obj in positions.keys() if "drawer" in obj]
        all_handles = [obj for obj in positions.keys() if "handle" in obj]
        all_locations = []
        for v in positions.values():
            if isinstance(v, (tuple, list)) and len(v) == 3:
                all_locations.append(tuple(v))
        # Remove duplicates
        all_locations = [x for i, x in enumerate(all_locations) if x not in all_locations[:i]]

        # For the purpose of primitive exploration - pick one object, one drawer, one handle, and their locations
        # (Fallback: Use first available names, unless you have better heuristics)
        some_object = all_objects[0] if all_objects else None
        some_drawer = all_drawers[0] if all_drawers else None
        some_handle = all_handles[0] if all_handles else None

        robot_pos = positions.get("robot", None)
        # Fallback: guess at robot name
        if robot_pos is None:
            keys = [k for k in positions.keys() if 'robot' in k or 'base' in k]
            if keys:
                robot_pos = positions[keys[0]]
        # If still unknown, choose a location as a fallback
        if robot_pos is None and all_locations:
            robot_pos = all_locations[0]
        
        # Confirm component availability
        print(f"[Exploration] Found object: {some_object}, drawer: {some_drawer}, handle: {some_handle}, robot pos: {robot_pos}")

        # 1. Try to move the robot (execute_go) to another location if possible
        explored = False
        move_targets = [loc for loc in all_locations if loc != robot_pos]
        if robot_pos and move_targets:
            to_pos = move_targets[0]
            print(f"[Exploration] Trying execute_go from {robot_pos} to {to_pos}")
            try:
                obs, reward, done = execute_go(env, task, robot_pos, to_pos)
                explored = True
                print(f"[Exploration] execute_go succeeded: robot moved to {to_pos}")
                robot_pos = to_pos
            except Exception as e:
                print(f"[Exploration] execute_go failed: {e}")

        # 2. Try picking an object on the floor (execute_pick)
        if some_object and robot_pos:
            print(f"[Exploration] Trying execute_pick for object {some_object} at location {robot_pos}")
            try:
                obs, reward, done = execute_pick(env, task, some_object, robot_pos)
                explored = True
                print(f"[Exploration] execute_pick succeeded: picked {some_object}")
            except Exception as e:
                print(f"[Exploration] execute_pick failed: {e}")

        # 3. Try opening (pulling) a drawer with a handle (requires robot holding handle)
        # First, pick the handle if any
        if some_handle and robot_pos:
            print(f"[Exploration] Trying execute_pick for handle {some_handle} at {robot_pos}")
            try:
                obs, reward, done = execute_pick(env, task, some_handle, robot_pos)
                print(f"[Exploration] Picked handle {some_handle}, now trying to execute_pull")
                if some_drawer:
                    try:
                        obs, reward, done = execute_pull(env, task, some_drawer, some_handle, robot_pos)
                        explored = True
                        print(f"[Exploration] execute_pull succeeded: drawer {some_drawer} manipulated with handle {some_handle}")
                    except Exception as e:
                        print(f"[Exploration] execute_pull failed: {e}")
            except Exception as e:
                print(f"[Exploration] execute_pick (handle) failed: {e}")

        # 4. Try placing an object in a drawer (requires: holding object, drawer open - usually done after exploration above)
        # Check if some object is held (might need to call execute_pick again)
        # If opened, attempt execute_place
        if some_object and some_drawer and robot_pos:
            print(f"[Exploration] Trying execute_place for {some_object} into {some_drawer} at {robot_pos}")
            try:
                obs, reward, done = execute_place(env, task, some_object, some_drawer, robot_pos)
                explored = True
                print(f"[Exploration] execute_place attempted.")
            except Exception as e:
                print(f"[Exploration] execute_place failed: {e}")

        # 5. Try pushing the drawer closed
        if some_drawer and robot_pos:
            print(f"[Exploration] Trying execute_push to close {some_drawer} at {robot_pos}")
            try:
                obs, reward, done = execute_push(env, task, some_drawer, robot_pos)
                explored = True
                print(f"[Exploration] execute_push attempted!")
            except Exception as e:
                print(f"[Exploration] execute_push failed: {e}")

        # 6. Try "gripper" and "sweep" if needed for further predicate identification
        print(f"[Exploration] Trying execute_gripper and execute_sweep for feature probing (if supported).")
        try:
            obs, reward, done = execute_gripper(env, task)
        except Exception as e:
            print(f"[Exploration] execute_gripper failed: {e}")

        if some_object and robot_pos:
            try:
                obs, reward, done = execute_sweep(env, task, some_object, robot_pos)
            except Exception as e:
                print(f"[Exploration] execute_sweep failed: {e}")

        print("[Exploration] Finished primitive exploration. Check logs or state diffs for discovered missing predicates or unexplained preconditions.")

        # --------------------------------------------------------
        # End of the exploration stage for missing predicates.
        # --------------------------------------------------------

        # If a concrete oracle plan is given later or found, add step-by-step execution here
        # using the provided skills, e.g.:
        # obs, reward, done = execute_pick(env, task, ...)
        # Repeat for each action in the proper (oracle) plan once the domain and problem are valid.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
