# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only externally provided skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, now with exploration to find missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for simulation capture
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object and Location Positions ===
        positions = get_object_positions()

        # === Exploration to Find the Missing Predicate (Based on Feedback) ===
        # Feedback: (holding handle_bottom) -- we must identify what predicate (e.g., lock-known, etc.) is missing

        # -- Step 1: Identify relevant objects and handles --
        # We'll look for any objects called 'handle', 'handle_bottom', or similar
        handle_name = None
        location_name = None
        robot_location = None
        # Find handle and the associated location
        for obj in positions:
            if 'handle' in obj:
                handle_name = obj
                # Also pick its location if available (from positions dict)
                if isinstance(positions[obj], dict) and "location" in positions[obj]:
                    location_name = positions[obj]["location"]
                else:
                    # Fallback: Use the first available location
                    continue
        # Fallback if no location is directly associated:
        if location_name is None:
            for obj in positions:
                if "location" in obj:
                    location_name = obj
                    break

        # Find robot's current location
        for obj in positions:
            if "robot" in obj or "agent" in obj:
                robot_location = positions[obj]
                break
        if robot_location is None:
            # Try from observation descriptions
            for entry in descriptions:
                if "robot-at" in entry:
                    robot_location = entry.split()[-1].replace(")", "")
                    break

        # If not found, set default/fallback location
        if location_name is None:
            location_name = "drawer_area"
        if handle_name is None:
            handle_name = "handle_bottom"
        if robot_location is None:
            robot_location = "init"

        # --- Step 2: Exploration Plan Execution ---
        # We'll use available skills to simulate pulling the handle and observe any missing effects

        print(f"[Exploration] Robot location: {robot_location}, Handle: {handle_name}, Location: {location_name}")

        # Try to pick up the handle if possible
        try:
            print("[Exploration] Attempting to pick up the handle...")
            obs, reward, done = execute_pick(env, task, handle_name, location_name)
            if done:
                print(f"[Exploration] Successfully picked up: {handle_name}")
            else:
                print(f"[Exploration] Pick action did not finish as expected for: {handle_name}")
        except Exception as e:
            print(f"[Exploration] Exception during pick: {e}")

        # At this point, robot is 'holding handle_bottom' -- try exploration action 'execute_pull'
        # This might check for a missing predicate such as 'lock-known'

        # Try to pull the handle and see if exploration yields a 'lock-known' effect or not
        try:
            print("[Exploration] Attempting to pull the handle (to determine missing predicate)...")
            obs, reward, done = execute_pull(env, task, handle_name, location_name)
            if done:
                print(f"[Exploration] Pull action completed for {handle_name}.")
            else:
                print(f"[Exploration] Pull action did not finish as expected for {handle_name}.")
        except Exception as e:
            print(f"[Exploration] Exception during pull: {e}")

        # If pull fails or the effect is missing, it's likely that a predicate such as (lock-known handle_bottom)
        # or similar needs to be set/checked explicitly. According to your exploration domain,
        # execute_pull is used to determine lock-known by trying to pull 'handle_bottom' when it is being held.

        print("[Exploration] Exploration phase complete. Check logs or effects for missing predicate such as (lock-known handle_bottom), as suggested by feedback.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()