# run_skeleton_task.py (Executable exploration and plan code per instructions)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # use only provided skills; do not redefine primitives

from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, now with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers for capturing simulation (optional)
        init_video_writers(obs)

        # Wrap task steps for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # positions: dictionary mapping object/location names to their positions

        # ————————————————————————————————————————————
        # EXPLORATION PHASE: Find missing predicate/knowledge
        # ————————————————————————————————————————————
        # Feedback: (robot-at drawer-area)
        # The robot-at predicate is relevant for binding robot to a location (here, 'drawer-area')
        # In the main domain, the robot's location (robot-at ?p - location) is critical for skill preconditions.
        # However, if a missing predicate/binding causes problems in later plan execution, we should explore locations.

        # -- Try moving the robot to the 'drawer-area' to see if the missing predicate is resolved by this motion.
        # (Could also be used to implicitly trigger needed knowledge, as in exploration domain.)

        try:
            # Find the name of a location that is or corresponds to 'drawer-area'.
            drawer_area_name = None
            for loc_name in positions.keys():
                if 'drawer' in loc_name and 'area' in loc_name:
                    drawer_area_name = loc_name
                    break
            if drawer_area_name is None:
                # fallback: just take a location called 'drawer' if present
                for loc_name in positions.keys():
                    if 'drawer' in loc_name:
                        drawer_area_name = loc_name
                        break

            # If still not found, pick any location (to continue script)
            if drawer_area_name is None:
                locations = [k for k in positions.keys() if 'area' in k or 'room' in k or 'location' in k]
                if locations:
                    drawer_area_name = locations[0]
                else:
                    # Drastic fallback: pick first key
                    drawer_area_name = list(positions.keys())[0]

            print(f"[Exploration] Trying to move the robot to location: {drawer_area_name}")

            # Try using the provided skill 'execute_go' to move the robot to this area.
            # In primitive_skills_static_lock_v2, execute_go takes parameters (?from ?to - location)
            # We must determine robot's current location. We'll check for it in positions dict, else set a default.

            # Find robot's current (likely initial) position; typically specified or can be guessed.
            # Assume that in get_object_positions, there's either a 'robot' or 'robot-at' entry, or infer from context.

            robot_location = None
            for name in positions.keys():
                if name.startswith('robot-at_') or name.startswith('robot_') or name == 'robot':
                    robot_location = name
                    break
            # If not found, find a plausible initial location
            if robot_location is None:
                for name in positions.keys():
                    if 'area' in name or 'room' in name or 'location' in name:
                        robot_location = name
                        break
            if robot_location is None:
                robot_location = list(positions.keys())[0]

            # Call the 'execute_go' skill to move from current location to drawer-area
            print(f"[Exploration] Robot at '{robot_location}' -- moving to '{drawer_area_name}'")
            try:
                # execute_go(env, task, from_location, to_location)
                obs, reward, done, info = execute_go(env, task, robot_location, drawer_area_name)
                print("[Exploration] execute_go to drawer-area completed.")
            except Exception as e:
                print(f"[Exploration] execute_go failed: {e}")

        except Exception as ex:
            print(f"[Exploration] Could not complete exploration phase: {ex}")

        # ————————————————————————————————
        # After exploration, proceed with normal plan / oracle plan
        # ————————————————————————————————
        # Here, in a real plan, you would sequence through e.g.:
        # - execute_pick
        # - execute_pull/push
        # - execute_place
        # using the locations, object names, drawer/handle names, etc., as required by the PDDL.
        # For this example, we will demonstrate placeholder calls for these skills where applicable.

        # Find objects, drawers, handles in positions/objects info:
        object_names = [k for k in positions.keys() if (('obj' in k) or ('ball' in k) or ('item' in k))]
        drawer_names = [k for k in positions.keys() if 'drawer' in k]
        handle_names = [k for k in positions.keys() if 'handle' in k]

        # Placeholders for actual names used by your environment
        obj_name = object_names[0] if object_names else None
        drawer_name = drawer_names[0] if drawer_names else None
        handle_name = handle_names[0] if handle_names else None

        # Robot is now at drawer-area. We'll try to pick an object, pull open the drawer, and place the object in it.
        # Note: Check and handle None before using objects.

        # Step 1: execute_pick (pick up object from floor)
        if obj_name is not None:
            try:
                print(f"[Task] Attempting to pick up object: {obj_name}")
                obs, reward, done, info = execute_pick(env, task, obj_name, drawer_area_name)
                print("[Task] execute_pick succeeded.")
            except Exception as e:
                print(f"[Task] execute_pick failed: {e}")
        else:
            print("[Task] No object found to pick up.")

        # Step 2: execute_pull (pull drawer open using handle)
        # execute_pull, as per domain, requires holding the handle-of the drawer
        if handle_name is not None and drawer_name is not None:
            try:
                print(f"[Task] Attempting to pick handle '{handle_name}' to open drawer '{drawer_name}'.")
                # Must pick the handle first (using execute_pick), then pull.
                obs, reward, done, info = execute_pick(env, task, handle_name, drawer_area_name)
                print(f"[Task] execute_pick (handle) succeeded. Now pulling drawer.")
                obs, reward, done, info = execute_pull(env, task, drawer_name, handle_name, drawer_area_name)
                print("[Task] execute_pull succeeded: Drawer should be open.")
            except Exception as e:
                print(f"[Task] Failed to open drawer with handle: {e}")
        else:
            print("[Task] Drawer or handle not found for pulling.")

        # Step 3: execute_place (place the object into the drawer)
        if obj_name is not None and drawer_name is not None:
            try:
                print(f"[Task] Attempting to place object '{obj_name}' into drawer '{drawer_name}'.")
                obs, reward, done, info = execute_place(env, task, obj_name, drawer_name, drawer_area_name)
                print("[Task] execute_place succeeded: Object should be in drawer.")
            except Exception as e:
                print(f"[Task] Failed to place object into drawer: {e}")
        else:
            print("[Task] No object or drawer for placement.")

        # Optionally: push drawer closed (execute_push)
        if drawer_name is not None:
            try:
                print(f"[Task] Attempting to push-close drawer '{drawer_name}'.")
                obs, reward, done, info = execute_push(env, task, drawer_name, drawer_area_name)
                print("[Task] execute_push succeeded: Drawer should be closed.")
            except Exception as e:
                print(f"[Task] Failed to close the drawer: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()