# run_skeleton_task.py (Filled in to perform predicate exploration as requested)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only the provided skill functions

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # e.g. positions = { 'drawer1': (x, y, z), 'obj1': (x, y, z), ... }
        positions = get_object_positions()

        # === EXPLORATION PHASE FOR MISSING PREDICATE ===
        # Based on Exploration Knowledge, the missing predicates are about knowledge:
        # identified, temperature-known, weight-known, durability-known, lock-known
        # Hence, we try to execute the available skills to probe and observe: e.g., pull/pick, etc.

        # The available skills in skill_code are:
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 
        #  'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # We'll attempt to manipulate all detectable objects to trigger informative state changes
        print("[Exploration] Attempting to execute all available skills on each object and drawer...")

        # Get possible object keys/types
        object_keys = [key for key in positions.keys() if 'obj' in key or 'object' in key]
        drawer_keys = [key for key in positions.keys() if 'drawer' in key]
        handle_keys = [key for key in positions.keys() if 'handle' in key]

        # For exploration, we iterate and attempt as many skills as possible
        for obj_name in object_keys:
            obj_pos = positions[obj_name]

            # Try execute_pick
            try:
                print(f"[Exploration] Trying execute_pick on {obj_name} at {obj_pos}")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    obj_name,
                    obj_pos
                )
            except Exception as e:
                print(f"[Exploration] execute_pick failed for {obj_name}: {e}")

            # If there are drawers available, try placing into each
            for drawer_name in drawer_keys:
                try:
                    print(f"[Exploration] Trying execute_place on {obj_name} into {drawer_name} at {obj_pos}")
                    obs, reward, done = execute_place(
                        env,
                        task,
                        obj_name,
                        drawer_name,
                        obj_pos
                    )
                except Exception as e:
                    print(f"[Exploration] execute_place failed for {obj_name} into {drawer_name}: {e}")

            # Try execute_sweep
            try:
                print(f"[Exploration] Trying execute_sweep on {obj_name} at {obj_pos}")
                obs, reward, done = execute_sweep(
                    env,
                    task,
                    obj_name,
                    obj_pos
                )
            except Exception as e:
                print(f"[Exploration] execute_sweep failed for {obj_name}: {e}")

        # Try to interact with each drawer: open (pull), close (push), and rotate/gripper
        for drawer_name in drawer_keys:
            drawer_pos = positions[drawer_name]
            # Find handle for this drawer
            handle_candidates = [hn for hn in handle_keys if hn in drawer_name or drawer_name in hn]
            handle_name = handle_candidates[0] if handle_candidates else None

            # Try opening drawer (requires a handle)
            if handle_name:
                try:
                    print(f"[Exploration] Trying execute_pick (for handle) on {handle_name} at {drawer_pos}")
                    obs, reward, done = execute_pick(
                        env,
                        task,
                        handle_name,
                        drawer_pos
                    )
                    print(f"[Exploration] Trying execute_pull on {drawer_name} with handle {handle_name} at {drawer_pos}")
                    obs, reward, done = execute_pull(
                        env,
                        task,
                        drawer_name,
                        handle_name,
                        drawer_pos
                    )
                except Exception as e:
                    print(f"[Exploration] execute_pull failed for {drawer_name} (handle: {handle_name}): {e}")

            # Try pushing drawer closed
            try:
                print(f"[Exploration] Trying execute_push on {drawer_name} at {drawer_pos}")
                obs, reward, done = execute_push(
                    env,
                    task,
                    drawer_name,
                    drawer_pos
                )
            except Exception as e:
                print(f"[Exploration] execute_push failed for {drawer_name}: {e}")

        # Try moving the robot to various locations using execute_go
        location_keys = [key for key in positions.keys() if 'location' in key or 'loc' in key or 'room' in key]
        # For demonstration, if at least two locations are available, try moving between them
        if len(location_keys) >= 2:
            from_loc, to_loc = location_keys[:2]
            try:
                print(f"[Exploration] Trying execute_go from {from_loc} to {to_loc}")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_loc,
                    to_loc
                )
            except Exception as e:
                print(f"[Exploration] execute_go failed from {from_loc} to {to_loc}: {e}")

        # Try gripper and rotate skills, if available
        try:
            print("[Exploration] Trying execute_gripper")
            obs, reward, done = execute_gripper(
                env,
                task
            )
        except Exception as e:
            print(f"[Exploration] execute_gripper failed: {e}")

        try:
            print("[Exploration] Trying execute_rotate")
            obs, reward, done = execute_rotate(
                env,
                task
            )
        except Exception as e:
            print(f"[Exploration] execute_rotate failed: {e}")

        # Summarize: user can now inspect the logs and simulation state to identify any missing predicates
        print("[Exploration] Exploration phase complete. Check logs/CUI for observed predicates or missing information.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()