# run_skeleton_task.py

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        try:
            positions = get_object_positions()
        except Exception as e:
            print("[Error] Could not get object positions:", e)
            positions = {}

        # --- EXPLORATION PHASE: Try available skills to diagnose missing predicate/step ---

        feedback_missing_predicate = False

        # The exploration domain reveals missing "lock-known" predicate may be needed before actions on drawers.

        # Since we don't know which predicate is missing (it caused previous planning to timeout), let's explore by trying to execute all relevant skills with available objects and locations.
        # We assume object and drawer names can be found from observation descriptions or object_positions().

        print("[Exploration] Starting exploration to diagnose missing predicates, e.g., 'lock-known'.")

        # List of available skill names
        skill_names = ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # Try to extract candidate objects and locations from object_positions or descriptions
        object_names = []
        drawer_names = []
        handle_names = []
        location_names = []
        for name in positions:
            if 'drawer' in name:
                drawer_names.append(name)
            elif 'handle' in name:
                handle_names.append(name)
            elif 'obj' in name or 'block' in name or 'ball' in name or 'cube' in name:
                object_names.append(name)
            elif 'table' in name or 'floor' in name or 'surface' in name or 'loc' in name:
                location_names.append(name)
        # Also try from descriptions if positions are empty/missing
        if not (object_names or drawer_names or location_names):
            if 'objects' in descriptions:
                # e.g., "objects": ["obj1", "drawer1", "loc1"]
                for name in descriptions['objects']:
                    if 'drawer' in name:
                        drawer_names.append(name)
                    elif 'handle' in name:
                        handle_names.append(name)
                    elif 'obj' in name or 'block' in name or 'ball' in name or 'cube' in name:
                        object_names.append(name)
                    elif 'table' in name or 'floor' in name or 'surface' in name or 'loc' in name:
                        location_names.append(name)

        # Fallback location candidate
        if not location_names:
            location_names = ['table', 'floor', 'loc1', 'loc2']

        # Fallback object/drawer/handle
        if not object_names:
            object_names = ['obj1', 'cube1']
        if not drawer_names:
            drawer_names = ['drawer1']
        if not handle_names:
            handle_names = ['handle1']

        # For the robot's starting position (if not available, use first location in list)
        robot_location = None
        for k, v in positions.items():
            if 'robot' in k and isinstance(v, str):
                robot_location = v
                break
        if not robot_location:
            robot_location = location_names[0]

        # === SYSTEMATIC EXPLORATION OF SKILLS ===
        for skill in skill_names:
            try:
                print(f"[Exploration] Trying skill: {skill}")
                # For each skill, try a variety of argument combinations
                # -- execute_pick: (object, location)
                if skill == 'execute_pick':
                    for o in object_names:
                        for p in location_names:
                            try:
                                print(f"  Attempting execute_pick({o}, {p})")
                                obs, reward, done = execute_pick(env, task, obj=o, location=p)
                                if done:
                                    print("[Exploration] Task ended after execute_pick.")
                                    return
                            except Exception as ex:
                                print(f"    [Fail] execute_pick({o}, {p}):", ex)
                # -- execute_place: (object, drawer, location)
                elif skill == 'execute_place':
                    for o in object_names:
                        for d in drawer_names:
                            for p in location_names:
                                try:
                                    print(f"  Attempting execute_place({o}, {d}, {p})")
                                    obs, reward, done = execute_place(env, task, obj=o, drawer=d, location=p)
                                    if done:
                                        print("[Exploration] Task ended after execute_place.")
                                        return
                                except Exception as ex:
                                    print(f"    [Fail] execute_place({o}, {d}, {p}):", ex)
                # -- execute_push: (drawer, location)
                elif skill == 'execute_push':
                    for d in drawer_names:
                        for p in location_names:
                            try:
                                print(f"  Attempting execute_push({d}, {p})")
                                obs, reward, done = execute_push(env, task, drawer=d, location=p)
                                if done:
                                    print("[Exploration] Task ended after execute_push.")
                                    return
                            except Exception as ex:
                                print(f"    [Fail] execute_push({d}, {p}):", ex)
                # -- execute_pull: (drawer, handle, location)
                elif skill == 'execute_pull':
                    for d in drawer_names:
                        for h in handle_names:
                            for p in location_names:
                                try:
                                    print(f"  Attempting execute_pull({d}, {h}, {p})")
                                    obs, reward, done = execute_pull(env, task, drawer=d, handle=h, location=p)
                                    if done:
                                        print("[Exploration] Task ended after execute_pull.")
                                        return
                                except Exception as ex:
                                    print(f"    [Fail] execute_pull({d}, {h}, {p}):", ex)
                # -- execute_sweep: (object, location)
                elif skill == 'execute_sweep':
                    for o in object_names:
                        for p in location_names:
                            try:
                                print(f"  Attempting execute_sweep({o}, {p})")
                                obs, reward, done = execute_sweep(env, task, obj=o, location=p)
                                if done:
                                    print("[Exploration] Task ended after execute_sweep.")
                                    return
                            except Exception as ex:
                                print(f"    [Fail] execute_sweep({o}, {p}):", ex)
                # -- execute_rotate: (object, location)
                elif skill == 'execute_rotate':
                    for o in object_names:
                        for p in location_names:
                            try:
                                print(f"  Attempting execute_rotate({o}, {p})")
                                obs, reward, done = execute_rotate(env, task, obj=o, location=p)
                                if done:
                                    print("[Exploration] Task ended after execute_rotate.")
                                    return
                            except Exception as ex:
                                print(f"    [Fail] execute_rotate({o}, {p}):", ex)
                # -- execute_go: (from_loc, to_loc)
                elif skill == 'execute_go':
                    for fr in location_names:
                        for to in location_names:
                            if fr != to:
                                try:
                                    print(f"  Attempting execute_go({fr}, {to})")
                                    obs, reward, done = execute_go(env, task, from_location=fr, to_location=to)
                                    if done:
                                        print("[Exploration] Task ended after execute_go.")
                                        return
                                except Exception as ex:
                                    print(f"    [Fail] execute_go({fr}, {to}):", ex)
                # -- execute_gripper: ()
                elif skill == 'execute_gripper':
                    try:
                        print(f"  Attempting execute_gripper()")
                        obs, reward, done = execute_gripper(env, task)
                        if done:
                            print("[Exploration] Task ended after execute_gripper.")
                            return
                    except Exception as ex:
                        print(f"    [Fail] execute_gripper():", ex)

            except Exception as e:
                print("[Exploration Error]", skill, e)
        
        print("[Exploration] Completed exhaustive skill exploration.")

        # Summary
        print("[Exploration] If none of the primitive skills executed successfully or the planning system still reports a missing predicate/timeout,")
        print("then it is likely that a predicate such as 'lock-known' (as highlighted in the exploration domain knowledge) is required by the plan/domain.")
        print("Please update your domain/problem files or obtain 'lock-known' information before retrying oracle plan execution.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
