# run_skeleton_task.py (Executable with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use the primitives provided; do not redefine them

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === BEGIN EXPLORATION FOR MISSING PREDICATES ===
        print("[Exploration] Starting exploration to discover missing predicates or object properties.")

        # Identify available keys (example: object names and locations)
        try:
            object_names = [k for k in positions.keys() if 'object' in k or 'obj' in k]
            location_names = [k for k in positions.keys() if 'location' in k or 'drawer' in k or 'room' in k or 'table' in k]
            handle_names = [k for k in positions.keys() if 'handle' in k]
        except Exception as e:
            print(f"[Exploration] Error extracting object/location names: {e}")
            object_names = []
            location_names = []
            handle_names = []

        # Try to explore all predicates by attempting to execute each available skill if possible
        # Use only predefined skills, according to available skill names
        available_skills = ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # Exploration: Try to use each available skill with guessed parameters to probe environment
        for skill in available_skills:
            try:
                print(f"[Exploration] Attempting skill: {skill}")

                if skill == 'execute_go':
                    # Try moving between two locations
                    if len(location_names) >= 2:
                        from_loc = location_names[0]
                        to_loc = location_names[1]
                        print(f"  [Exploration] Trying execute_go from {from_loc} to {to_loc}")
                        obs, reward, done = execute_go(env, task, from_loc, to_loc)
                        if done:
                            print("[Exploration] Task ended after execute_go!")
                            break
                        
                elif skill == 'execute_pick':
                    # Try picking up an object
                    if len(object_names) > 0 and len(location_names) > 0:
                        obj = object_names[0]
                        loc = location_names[0]
                        print(f"  [Exploration] Trying execute_pick {obj} at {loc}")
                        obs, reward, done = execute_pick(env, task, obj, loc)
                        if done:
                            print("[Exploration] Task ended after execute_pick!")
                            break

                elif skill == 'execute_pull':
                    # Try pulling a handle if any
                    # Need (drawer, handle, location); handle-of relation not known here, so guess
                    if len(location_names) > 0 and len(handle_names) > 0:
                        drawer = location_names[0]
                        handle = handle_names[0]
                        loc = location_names[0]
                        print(f"  [Exploration] Trying execute_pull {drawer}, {handle}, {loc}")
                        obs, reward, done = execute_pull(env, task, drawer, handle, loc)
                        if done:
                            print("[Exploration] Task ended after execute_pull!")
                            break

                elif skill == 'execute_place':
                    # Try to place an object in a drawer/location if possible
                    if len(object_names) > 0 and len(location_names) > 1:
                        obj = object_names[0]
                        drawer = location_names[0]
                        loc = location_names[1]
                        print(f"  [Exploration] Trying execute_place {obj}, {drawer}, {loc}")
                        obs, reward, done = execute_place(env, task, obj, drawer, loc)
                        if done:
                            print("[Exploration] Task ended after execute_place!")
                            break

                elif skill == 'execute_push':
                    # Try to push (close) a drawer
                    if len(location_names) > 0:
                        drawer = location_names[0]
                        loc = location_names[0]
                        print(f"  [Exploration] Trying execute_push {drawer}, {loc}")
                        obs, reward, done = execute_push(env, task, drawer, loc)
                        if done:
                            print("[Exploration] Task ended after execute_push!")
                            break

                elif skill == 'execute_sweep':
                    # Try sweeping an object at a location
                    if len(object_names) > 0 and len(location_names) > 0:
                        obj = object_names[0]
                        loc = location_names[0]
                        print(f"  [Exploration] Trying execute_sweep {obj}, {loc}")
                        obs, reward, done = execute_sweep(env, task, obj, loc)
                        if done:
                            print("[Exploration] Task ended after execute_sweep!")
                            break

                elif skill == 'execute_rotate':
                    # rotate is not defined in original domain, likely a placeholder; skip unless implemented
                    print(f"  [Exploration] Skipping execute_rotate (not implemented in domain).")

                elif skill == 'execute_gripper':
                    # execute_gripper has no parameters; try to execute
                    print(f"  [Exploration] Trying execute_gripper()")
                    obs, reward, done = execute_gripper(env, task)
                    if done:
                        print("[Exploration] Task ended after execute_gripper!")
                        break

                else:
                    print(f"  [Exploration] Unknown skill: {skill}")
            except Exception as ex:
                print(f"  [Exploration] Skill {skill} raised exception: {ex}")

        print("[Exploration] Exploration phase completed. Please analyze logs for missing predicate clues.")

        # === END EXPLORATION ===

        # (Optional) Here you would execute the oracle plan step-by-step with available skills
        print("[Task] You can now proceed to perform the planned sequence of actions using your oracle plan.")

        # TODO: Here, insert action calls according to the oracle plan after missing predicates are discovered.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
