# run_skeleton_task.py (Completed for Exploration of Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicate.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # ----------- Exploration Phase: Find Missing Predicate -----------
        # Feedback: (on-floor drawer_top_handle)
        # Based on this, we need to determine if the object 'drawer_top_handle' is actually classified as on the floor.

        # Let's get all object keys and look for handle-related objects
        drawer_top_handle = None
        for obj_name in positions:
            if "handle" in obj_name and ("top" in obj_name or "drawer" in obj_name):
                drawer_top_handle = obj_name
                break

        if drawer_top_handle is None:
            print("[Exploration] Could not find 'drawer_top_handle' in object positions.")
        else:
            print(f"[Exploration] Found potential handle object: {drawer_top_handle}")

            # Let's check if our predicates/skills can handle "on-floor drawer_top_handle"
            # We'll try to execute relevant skills and observe any failure/feedback.

            # 1. Try to pick the drawer_top_handle (simulate explore-on-floor-pickable property)
            try:
                # Find location for handle object
                handle_pos = positions[drawer_top_handle]

                # Attempt to use execute_pick (expects object to be on the floor)
                # We assume robot is already free, hand is empty, and at correct location.
                # If not, we may need to call 'execute_go' to go to the handle's location.

                robot_location = None
                for obj_name, pos in positions.items():
                    if "robot" in obj_name:
                        robot_location = obj_name
                        break

                # We'll just use the location associated with the handle
                handle_location = None
                if hasattr(handle_pos, "__len__") and len(handle_pos) == 3:
                    handle_location = drawer_top_handle + "_location"
                else:
                    # Or fallback to a generic location string if not provided
                    handle_location = "handle_location"

                print(f"[Exploration] Attempting to pick {drawer_top_handle} at location {handle_location}...")

                # Move robot (if required)
                try:
                    obs, reward, done = execute_go(
                        env,
                        task,
                        from_location="robot_init",   # Use a placeholder/start location
                        to_location=handle_location
                    )
                except Exception as e:
                    print(f"[Exploration] execute_go failed (may be at location already): {e}")

                # Try picking the handle as an object "on-floor"
                try:
                    obs, reward, done = execute_pick(
                        env,
                        task,
                        object_name=drawer_top_handle,
                        location=handle_location
                    )
                    print(f"[Exploration] execute_pick succeeded for {drawer_top_handle}.")
                except Exception as e:
                    print(f"[Exploration] execute_pick failed for {drawer_top_handle}: {e}")
                    print("[Exploration] This may indicate 'on-floor' predicate is missing for handle.")

                    # If picking isn't possible, try using another skill to explore further
                    # For example, try execute_sweep, execute_gripper, or execute_pull if meaningful

                # If the pick succeeds, we've confirmed that the handle is "on-floor",
                # If not, the missing predicate may be that handles should be modelled with (on-floor ...)

            except Exception as err:
                print(f"[Exploration] Error examining on-floor status for handle: {err}")

        # ---- End of Exploration Phase ----

        # Here, you would continue with the remaining high-level plan (oracle plan),
        # using the predefined skills as needed, e.g.:
        #
        # obs, reward, done = execute_pick(env, task, object_name, location)
        # obs, reward, done = execute_place(env, task, object_name, drawer_name, location)
        # obs, reward, done = execute_pull(env, task, drawer_name, handle_object_name, location)
        # ...
        #
        # For this exploration task, we focus on detecting the missing predicate using only allowed skills.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
