# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is causing planning to fail.
        # We use the available skills to explore the environment and infer missing predicates.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.
        # We will attempt to use the available skills to interact with objects/drawers and observe the environment's response.

        # List of available skills
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # Try to explore all objects and drawers to find missing predicates
        # We assume positions contains keys for objects, drawers, handles, and locations
        # For demonstration, we use generic names; in practice, use actual keys from positions

        # Example keys (replace with actual keys from your environment)
        objects = [k for k in positions if k.startswith('object')]
        drawers = [k for k in positions if k.startswith('drawer')]
        handles = [k for k in positions if k.startswith('handle')]
        locations = [k for k in positions if k.startswith('location') or k.startswith('room')]

        # If not found, try to infer from all keys
        if not objects:
            objects = [k for k in positions if 'object' in k]
        if not drawers:
            drawers = [k for k in positions if 'drawer' in k]
        if not handles:
            handles = [k for k in positions if 'handle' in k]
        if not locations:
            locations = [k for k in positions if 'location' in k or 'room' in k]

        # For exploration, try to pick, pull, and open drawers to see if any action fails due to missing predicates
        # This will help us identify which predicate is missing

        exploration_done = False
        for obj in objects:
            for loc in locations:
                try:
                    print(f"[Exploration] Trying to pick {obj} at {loc}")
                    obs, reward, done = execute_pick(env, task, obj, loc)
                    if done:
                        print(f"[Exploration] Successfully picked {obj} at {loc}")
                        exploration_done = True
                        break
                except Exception as e:
                    print(f"[Exploration] execute_pick failed for {obj} at {loc}: {e}")
            if exploration_done:
                break

        if not exploration_done:
            for drawer in drawers:
                for handle in handles:
                    for loc in locations:
                        try:
                            print(f"[Exploration] Trying to pull {drawer} with {handle} at {loc}")
                            obs, reward, done = execute_pull(env, task, drawer, handle, loc)
                            if done:
                                print(f"[Exploration] Successfully pulled {drawer} with {handle} at {loc}")
                                exploration_done = True
                                break
                        except Exception as e:
                            print(f"[Exploration] execute_pull failed for {drawer} with {handle} at {loc}: {e}")
                    if exploration_done:
                        break
                if exploration_done:
                    break

        if not exploration_done:
            for from_loc in locations:
                for to_loc in locations:
                    if from_loc == to_loc:
                        continue
                    try:
                        print(f"[Exploration] Trying to go from {from_loc} to {to_loc}")
                        obs, reward, done = execute_go(env, task, from_loc, to_loc)
                        if done:
                            print(f"[Exploration] Successfully moved from {from_loc} to {to_loc}")
                            exploration_done = True
                            break
                    except Exception as e:
                        print(f"[Exploration] execute_go failed from {from_loc} to {to_loc}: {e}")
                if exploration_done:
                    break

        # After exploration, print a message if no action succeeded
        if not exploration_done:
            print("[Exploration] No skill action succeeded. The missing predicate may be blocking all actions.")

        # === Main Task Plan Execution ===
        # After exploration, proceed with the oracle plan if available.
        # For demonstration, we show a generic plan using available skills and positions.

        # Example: Move to a location, pick an object, open a drawer, place the object, close the drawer
        try:
            # Move to the first location
            if locations:
                from_loc = locations[0]
                to_loc = locations[1] if len(locations) > 1 else locations[0]
                print(f"[Task] Moving from {from_loc} to {to_loc}")
                obs, reward, done = execute_go(env, task, from_loc, to_loc)
                if done:
                    print("[Task] Reached target location.")

            # Pick the first object at the current location
            if objects and locations:
                obj = objects[0]
                loc = to_loc
                print(f"[Task] Picking {obj} at {loc}")
                obs, reward, done = execute_pick(env, task, obj, loc)
                if done:
                    print(f"[Task] Picked {obj}.")

            # Open the first drawer using the first handle at the current location
            if drawers and handles and locations:
                drawer = drawers[0]
                handle = handles[0]
                loc = to_loc
                print(f"[Task] Pulling {drawer} with {handle} at {loc}")
                obs, reward, done = execute_pull(env, task, drawer, handle, loc)
                if done:
                    print(f"[Task] Opened {drawer}.")

            # Place the object in the drawer
            if objects and drawers and locations:
                obj = objects[0]
                drawer = drawers[0]
                loc = to_loc
                print(f"[Task] Placing {obj} in {drawer} at {loc}")
                obs, reward, done = execute_place(env, task, obj, drawer, loc)
                if done:
                    print(f"[Task] Placed {obj} in {drawer}.")

            # Close the drawer
            if drawers and locations:
                drawer = drawers[0]
                loc = to_loc
                print(f"[Task] Pushing {drawer} at {loc}")
                obs, reward, done = execute_push(env, task, drawer, loc)
                if done:
                    print(f"[Task] Closed {drawer}.")

        except Exception as e:
            print(f"[Task] Exception during main plan execution: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()