# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Object and Drawer Names (from feedback and observation) ===
        # Ensure dice1 and dice2 are included in the object_names list
        object_names = [
            "dice1", "dice2", "handle1", "handle2", "handle3",
            "drawer1", "drawer2", "drawer3",
            "drawer_top_handle", "drawer_middle_handle", "drawer_bottom_handle",
            "drawer_top_place_left", "drawer_top_place_right", "drawer_middle_place",
            "drawer_bottom_place_left", "drawer_bottom_place_right"
        ]
        drawer_names = ["drawer1", "drawer2", "drawer3"]
        handle_names = ["handle1", "handle2", "handle3"]

        # === Initial State (from observation) ===
        # dice1, dice2, handle1, handle2, handle3 are on the floor
        # drawer1 is open and empty, unlocked
        # drawer2 is closed, empty, locked
        # drawer3 is closed, empty, unlocked
        # robot is at "ready-pose", hand is empty, robot is free

        # === Exploration Phase: Identify Missing Predicate ===
        # The task requires an exploration phase to determine which predicate is missing.
        # For this, we will attempt to interact with dice1 and dice2, and check if they are recognized.
        # If not, we will print a warning and skip the action.

        # Check that dice1 and dice2 exist in the scene
        missing_objects = []
        for obj in ["dice1", "dice2"]:
            if obj not in positions:
                print(f"[Exploration] WARNING: Object '{obj}' not found in scene positions.")
                missing_objects.append(obj)
        if missing_objects:
            print(f"[Exploration] The following objects are missing and cannot be manipulated: {missing_objects}")
            # Optionally, abort or continue with available objects

        # === Main Task Plan ===
        # Goal: Put dice1 and dice2 into the already open drawer1

        # 1. For each dice (dice1, dice2):
        #    a. Move to the dice location if not already there
        #    b. Pick the dice
        #    c. Move to drawer1 (if not already there)
        #    d. Place the dice in drawer1

        # For this, we need to know the location of dice1, dice2, and drawer1
        # Assume positions dict provides this info

        # Helper: get location of an object
        def get_location_of(obj_name):
            pos = positions.get(obj_name)
            if pos is None:
                return None
            # If positions are mapped to location names, return that
            # Otherwise, return the position tuple
            return pos

        # Assume robot starts at "ready-pose"
        robot_location = "ready-pose"
        drawer1_location = get_location_of("drawer1")
        if drawer1_location is None:
            print("[Error] drawer1 location not found. Aborting task.")
            return

        # For each dice
        for dice in ["dice1", "dice2"]:
            if dice in missing_objects:
                print(f"[Task] Skipping {dice} as it is missing from the scene.")
                continue

            dice_location = get_location_of(dice)
            if dice_location is None:
                print(f"[Task] Location for {dice} not found. Skipping.")
                continue

            # a. Move to dice location if not already there
            if robot_location != dice_location:
                try:
                    print(f"[Task] Moving to {dice} at location {dice_location}")
                    obs, reward, done, info = execute_go(env, task, robot_location, dice_location)
                    robot_location = dice_location
                except Exception as e:
                    print(f"[Error] Failed to move to {dice}: {e}")
                    continue

            # b. Pick the dice
            try:
                print(f"[Task] Picking up {dice}")
                obs, reward, done, info = execute_pick(env, task, dice, dice_location)
            except Exception as e:
                print(f"[Error] Failed to pick {dice}: {e}")
                continue

            # c. Move to drawer1 location if not already there
            if robot_location != drawer1_location:
                try:
                    print(f"[Task] Moving to drawer1 at location {drawer1_location}")
                    obs, reward, done, info = execute_go(env, task, robot_location, drawer1_location)
                    robot_location = drawer1_location
                except Exception as e:
                    print(f"[Error] Failed to move to drawer1: {e}")
                    continue

            # d. Place the dice in drawer1
            try:
                print(f"[Task] Placing {dice} in drawer1")
                obs, reward, done, info = execute_place(env, task, dice, "drawer1", drawer1_location)
            except Exception as e:
                print(f"[Error] Failed to place {dice} in drawer1: {e}")
                continue

        # === End of Main Task Plan ===

        print("[Task] Finished putting dice in drawer1.")

        # === Exploration: Check for missing predicates ===
        # If the above actions failed due to missing predicates (e.g., on-floor, in-drawer, etc.),
        # print a message to help identify which predicate is missing.
        # This is a placeholder for more advanced exploration logic.

        # For demonstration, check if both dice are now in drawer1
        # (Assume we can query the environment for this)
        try:
            dice1_in_drawer = False
            dice2_in_drawer = False
            # If the environment provides a way to check predicates, use it here
            # For now, just print a message
            print("[Exploration] (Simulated) Checking if dice1 and dice2 are in drawer1...")
            # If not, suggest that a predicate may be missing
            # (In a real system, you would query the environment state)
        except Exception as e:
            print(f"[Exploration] Could not verify dice in drawer1: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()