# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We use the available skills to explore the environment and infer missing predicates.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.
        # We will attempt to use the available skills to interact with objects/drawers and observe the environment's response.

        # Get all objects, drawers, and locations from positions
        # (Assume positions dict contains keys: 'objects', 'drawers', 'handles', 'locations')
        objects = positions.get('objects', [])
        drawers = positions.get('drawers', [])
        handles = positions.get('handles', [])
        locations = positions.get('locations', [])
        robot_location = positions.get('robot', None)

        # Helper: Move robot to a location using execute_go
        def move_robot(current_loc, target_loc):
            try:
                obs, reward, done = execute_go(env, task, from_location=current_loc, to_location=target_loc)
                return obs, reward, done
            except Exception as e:
                print(f"[Exploration] Error in execute_go: {e}")
                return None, 0, False

        # Helper: Try to pick up an object using execute_pick
        def try_pick(obj, loc):
            try:
                obs, reward, done = execute_pick(env, task, obj, loc)
                return obs, reward, done
            except Exception as e:
                print(f"[Exploration] Error in execute_pick: {e}")
                return None, 0, False

        # Helper: Try to open a drawer using execute_pull
        def try_pull(drawer, handle, loc):
            try:
                obs, reward, done = execute_pull(env, task, drawer, handle, loc)
                return obs, reward, done
            except Exception as e:
                print(f"[Exploration] Error in execute_pull: {e}")
                return None, 0, False

        # Helper: Try to close a drawer using execute_push
        def try_push(drawer, loc):
            try:
                obs, reward, done = execute_push(env, task, drawer, loc)
                return obs, reward, done
            except Exception as e:
                print(f"[Exploration] Error in execute_push: {e}")
                return None, 0, False

        # Helper: Try to place an object in a drawer using execute_place
        def try_place(obj, drawer, loc):
            try:
                obs, reward, done = execute_place(env, task, obj, drawer, loc)
                return obs, reward, done
            except Exception as e:
                print(f"[Exploration] Error in execute_place: {e}")
                return None, 0, False

        # Exploration: Try to interact with all objects and drawers to see if any action fails due to a missing predicate
        # This will help us infer which predicate is missing (e.g., lock-known, identified, etc.)

        # 1. Try to pick up each object from each location
        for obj in objects:
            for loc in locations:
                print(f"[Exploration] Attempting to pick {obj} at {loc}")
                obs, reward, done = try_pick(obj, loc)
                if done:
                    print(f"[Exploration] Successfully picked {obj} at {loc}")
                else:
                    print(f"[Exploration] Could not pick {obj} at {loc}")

        # 2. Try to open each drawer using its handle at each location
        for drawer in drawers:
            for handle in handles:
                for loc in locations:
                    print(f"[Exploration] Attempting to pull {drawer} with handle {handle} at {loc}")
                    obs, reward, done = try_pull(drawer, handle, loc)
                    if done:
                        print(f"[Exploration] Successfully pulled {drawer} with handle {handle} at {loc}")
                    else:
                        print(f"[Exploration] Could not pull {drawer} with handle {handle} at {loc}")

        # 3. Try to close each drawer at each location
        for drawer in drawers:
            for loc in locations:
                print(f"[Exploration] Attempting to push {drawer} at {loc}")
                obs, reward, done = try_push(drawer, loc)
                if done:
                    print(f"[Exploration] Successfully pushed {drawer} at {loc}")
                else:
                    print(f"[Exploration] Could not push {drawer} at {loc}")

        # 4. Try to place each object in each drawer at each location
        for obj in objects:
            for drawer in drawers:
                for loc in locations:
                    print(f"[Exploration] Attempting to place {obj} in {drawer} at {loc}")
                    obs, reward, done = try_place(obj, drawer, loc)
                    if done:
                        print(f"[Exploration] Successfully placed {obj} in {drawer} at {loc}")
                    else:
                        print(f"[Exploration] Could not place {obj} in {drawer} at {loc}")

        # 5. Try to sweep each object at each location
        for obj in objects:
            for loc in locations:
                print(f"[Exploration] Attempting to sweep {obj} at {loc}")
                try:
                    obs, reward, done = execute_sweep(env, task, obj, loc)
                    if done:
                        print(f"[Exploration] Successfully swept {obj} at {loc}")
                    else:
                        print(f"[Exploration] Could not sweep {obj} at {loc}")
                except Exception as e:
                    print(f"[Exploration] Error in execute_sweep: {e}")

        # 6. Try to use gripper (if applicable)
        try:
            print(f"[Exploration] Attempting to use gripper")
            obs, reward, done = execute_gripper(env, task)
            if done:
                print(f"[Exploration] Successfully used gripper")
            else:
                print(f"[Exploration] Could not use gripper")
        except Exception as e:
            print(f"[Exploration] Error in execute_gripper: {e}")

        # 7. Try to move robot between all pairs of locations
        for from_loc in locations:
            for to_loc in locations:
                if from_loc != to_loc:
                    print(f"[Exploration] Attempting to move from {from_loc} to {to_loc}")
                    obs, reward, done = move_robot(from_loc, to_loc)
                    if done:
                        print(f"[Exploration] Successfully moved from {from_loc} to {to_loc}")
                    else:
                        print(f"[Exploration] Could not move from {from_loc} to {to_loc}")

        # After exploration, analyze which actions consistently fail and check error messages.
        # This will help identify the missing predicate (e.g., lock-known, identified, etc.).
        # The user can then update the domain/problem accordingly.

        print("[Exploration] Exploration phase complete. Please review the above logs to identify missing predicates or preconditions.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()