# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We use the available skills to explore the environment and infer missing predicates.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.

        # We will attempt to use the available skills to interact with objects and drawers,
        # and observe if any action fails or if the environment provides feedback about missing knowledge.

        # Get all objects and locations from the positions dictionary
        object_names = [name for name in positions if positions[name].get('type', '') == 'object']
        drawer_names = [name for name in positions if positions[name].get('type', '') == 'drawer']
        location_names = [name for name in positions if positions[name].get('type', '') == 'location']

        # If type info is not available, fallback to all keys
        if not object_names:
            object_names = [name for name in positions if 'object' in name]
        if not drawer_names:
            drawer_names = [name for name in positions if 'drawer' in name]
        if not location_names:
            location_names = [name for name in positions if 'location' in name or 'room' in name or 'table' in name]

        # Try to move the robot to each location and interact with objects/drawers
        for loc in location_names:
            try:
                # Move robot to location (if not already there)
                # We need to know the current location; assume we start at the first location
                current_loc = location_names[0]
                if loc != current_loc:
                    print(f"[Exploration] Moving robot from {current_loc} to {loc}")
                    obs, reward, done = execute_go(env, task, current_loc, loc)
                    current_loc = loc

                # Try to pick up each object at this location
                for obj in object_names:
                    try:
                        print(f"[Exploration] Attempting to pick up {obj} at {loc}")
                        obs, reward, done = execute_pick(env, task, obj, loc)
                        if done:
                            print(f"[Exploration] Successfully picked up {obj} at {loc}")
                            # Try to place it in any drawer at this location
                            for drawer in drawer_names:
                                try:
                                    print(f"[Exploration] Attempting to place {obj} in {drawer} at {loc}")
                                    obs, reward, done = execute_place(env, task, obj, drawer, loc)
                                    if done:
                                        print(f"[Exploration] Successfully placed {obj} in {drawer} at {loc}")
                                        break
                                except Exception as e:
                                    print(f"[Exploration] Place failed for {obj} in {drawer} at {loc}: {e}")
                            break
                    except Exception as e:
                        print(f"[Exploration] Pick failed for {obj} at {loc}: {e}")

                # Try to open and close each drawer at this location
                for drawer in drawer_names:
                    # Try to find the handle for the drawer
                    handle_name = None
                    for obj in object_names:
                        if positions[obj].get('handle_of', '') == drawer:
                            handle_name = obj
                            break
                    if handle_name:
                        try:
                            print(f"[Exploration] Attempting to pick up handle {handle_name} at {loc}")
                            obs, reward, done = execute_pick(env, task, handle_name, loc)
                            if done:
                                print(f"[Exploration] Attempting to pull (open) drawer {drawer} with handle {handle_name} at {loc}")
                                obs, reward, done = execute_pull(env, task, drawer, handle_name, loc)
                                if done:
                                    print(f"[Exploration] Successfully opened drawer {drawer} at {loc}")
                                    # Try to close the drawer
                                    try:
                                        print(f"[Exploration] Attempting to push (close) drawer {drawer} at {loc}")
                                        obs, reward, done = execute_push(env, task, drawer, loc)
                                        if done:
                                            print(f"[Exploration] Successfully closed drawer {drawer} at {loc}")
                                    except Exception as e:
                                        print(f"[Exploration] Push failed for drawer {drawer} at {loc}: {e}")
                        except Exception as e:
                            print(f"[Exploration] Pull failed for drawer {drawer} with handle {handle_name} at {loc}: {e}")

            except Exception as e:
                print(f"[Exploration] Move failed to {loc}: {e}")

        # If any action fails with a message about a missing predicate, print it out
        print("[Exploration] If any action failed due to a missing predicate, check the error messages above.")
        print("[Exploration] Exploration phase complete. Use the above information to identify missing predicates in the domain/problem.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()