# run_skeleton_task.py (Completed for Exploration of Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # --- Exploration Phase: Find missing predicate (robot-at drawer-area) ---
        # The feedback indicates that the predicate (robot-at drawer-area) is missing.
        # We need to explore the environment to check if the robot can be at 'drawer-area'.
        # We'll use the available skills to move the robot to 'drawer-area' and observe the result.

        # Try to find a location named 'drawer-area' in the positions dictionary
        drawer_area_key = None
        for key in positions:
            if 'drawer' in key and 'area' in key:
                drawer_area_key = key
                break
        if drawer_area_key is None:
            # Fallback: try common variants
            for key in positions:
                if 'drawer' in key:
                    drawer_area_key = key
                    break

        if drawer_area_key is None:
            print("[Exploration] Could not find 'drawer-area' in object positions. Exploration aborted.")
            return

        # Find the robot's current location
        robot_location = None
        for key in positions:
            if 'robot' in key or 'base' in key:
                robot_location = key
                break
        if robot_location is None:
            # Fallback: try to infer from observation or use a default
            robot_location = 'start_area'
            if robot_location not in positions:
                # Just pick the first location as a last resort
                for key in positions:
                    robot_location = key
                    break

        print(f"[Exploration] Robot current location: {robot_location}")
        print(f"[Exploration] Target drawer area: {drawer_area_key}")

        # Use the available skill 'execute_go' to move the robot to the drawer area
        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_location,
                to_location=drawer_area_key,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print(f"[Exploration] Executed 'execute_go' from {robot_location} to {drawer_area_key}.")
        except Exception as e:
            print(f"[Exploration] Error during execute_go: {e}")
            return

        # After moving, check if the predicate (robot-at drawer-area) is now true
        # This can be done by checking the robot's position in the environment
        # (Assume get_object_positions() is updated after each step)
        positions = get_object_positions()
        robot_at_drawer_area = False
        if drawer_area_key in positions:
            # Compare robot's position to drawer_area position
            robot_pos = None
            drawer_area_pos = positions[drawer_area_key]
            for key in positions:
                if 'robot' in key or 'base' in key:
                    robot_pos = positions[key]
                    break
            if robot_pos is not None:
                # Use a simple distance threshold to check if robot is at drawer area
                dist = np.linalg.norm(np.array(robot_pos) - np.array(drawer_area_pos))
                if dist < 0.1:
                    robot_at_drawer_area = True

        if robot_at_drawer_area:
            print("[Exploration] Predicate (robot-at drawer-area) is TRUE after execute_go.")
        else:
            print("[Exploration] Predicate (robot-at drawer-area) is still missing or not satisfied.")

        # === End of Exploration Phase ===

        # (Optional) Continue with further plan steps if needed
        # For this task, the focus is on exploration and predicate checking.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()