# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback indicates (drawer-closed drawer1) is missing or needs to be checked.
        # We will use available skills to explore the state of drawer1.

        # Assumptions for object names and locations
        # You may need to adjust these names based on your environment
        drawer_name = 'drawer1'
        handle_name = 'handle1'
        robot_location = 'loc_robot'
        drawer_location = 'loc_drawer'
        # If your environment provides more precise names, use those.

        # Step 1: Move robot to drawer location if not already there
        try:
            robot_pos = positions.get('robot', None)
            drawer_pos = positions.get(drawer_name, None)
            if robot_pos is not None and drawer_pos is not None:
                if robot_pos != drawer_pos:
                    print(f"[Exploration] Moving robot from {robot_pos} to {drawer_pos}")
                    obs, reward, done = execute_go(env, task, robot_pos, drawer_pos)
                    if done:
                        print("[Exploration] Task ended during move!")
                        return
            else:
                print("[Exploration] Could not determine robot or drawer position, skipping move.")
        except Exception as e:
            print(f"[Exploration] Exception during move: {e}")

        # Step 2: Try to pull the drawer to check if it is closed/locked/unlocked
        # This will help us determine if (drawer-closed drawer1) is true/missing
        try:
            print(f"[Exploration] Attempting to pick handle {handle_name}")
            obs, reward, done = execute_pick(env, task, handle_name, drawer_location)
            if done:
                print("[Exploration] Task ended during handle pick!")
                return
        except Exception as e:
            print(f"[Exploration] Exception during handle pick: {e}")

        try:
            print(f"[Exploration] Attempting to pull drawer {drawer_name} using handle {handle_name}")
            obs, reward, done = execute_pull(env, task, drawer_name, handle_name, drawer_location)
            if done:
                print("[Exploration] Task ended during drawer pull!")
                return
        except Exception as e:
            print(f"[Exploration] Exception during drawer pull: {e}")
            print("[Exploration] This may indicate the drawer is locked or not closed.")

        # Step 3: If pull failed, try to push (close) the drawer to ensure it is closed
        try:
            print(f"[Exploration] Attempting to push (close) drawer {drawer_name}")
            obs, reward, done = execute_push(env, task, drawer_name, drawer_location)
            if done:
                print("[Exploration] Task ended during drawer push!")
                return
        except Exception as e:
            print(f"[Exploration] Exception during drawer push: {e}")

        # Step 4: Try pulling again after push to see if state changed
        try:
            print(f"[Exploration] Attempting to pull drawer {drawer_name} again after push")
            obs, reward, done = execute_pull(env, task, drawer_name, handle_name, drawer_location)
            if done:
                print("[Exploration] Task ended during drawer pull (after push)!")
                return
        except Exception as e:
            print(f"[Exploration] Exception during drawer pull (after push): {e}")

        print("[Exploration] Exploration phase complete. If all actions succeeded, (drawer-closed drawer1) is likely present and can be manipulated.")

        # === Main Task Plan (Example) ===
        # At this point, you can proceed with the main oracle plan using the available skills.
        # For example, pick an object and place it in the drawer, etc.
        # The following is a placeholder for your actual plan logic.

        # Example: Pick up an object and place it in the drawer
        object_to_pick = 'object1'
        object_location = 'loc_object1'
        try:
            print(f"[Task] Picking up {object_to_pick} at {object_location}")
            obs, reward, done = execute_pick(env, task, object_to_pick, object_location)
            if done:
                print("[Task] Task ended during object pick!")
                return
        except Exception as e:
            print(f"[Task] Exception during object pick: {e}")

        try:
            print(f"[Task] Placing {object_to_pick} in {drawer_name} at {drawer_location}")
            obs, reward, done = execute_place(env, task, object_to_pick, drawer_name, drawer_location)
            if done:
                print("[Task] Task ended during object place!")
                return
        except Exception as e:
            print(f"[Task] Exception during object place: {e}")

        print("[Task] Main task plan complete.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()