# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Object and Drawer Discovery ===
        # Try to robustly select two dice and a drawer from the environment
        # and check their compatibility and state

        # 1. Gather all objects and drawers
        object_names = list(positions.keys())
        dice_names = [name for name in object_names if "dice" in name.lower()]
        drawer_names = [name for name in object_names if "drawer" in name.lower()]
        handle_names = [name for name in object_names if "handle" in name.lower()]

        # 2. Select two dice
        selected_dice = []
        for dice in ["dice1", "dice2"]:
            if dice in dice_names:
                selected_dice.append(dice)
        # If not found by name, just pick first two dice-like objects
        if len(selected_dice) < 2:
            for name in dice_names:
                if name not in selected_dice and len(selected_dice) < 2:
                    selected_dice.append(name)
        if len(selected_dice) < 2:
            print("[Error] Not enough dice found in the environment. Aborting.")
            return

        # 3. Select a drawer and its handle
        if not drawer_names:
            print("[Error] No drawer found in the environment. Aborting.")
            return
        drawer = drawer_names[0]
        # Try to find a handle for this drawer
        handle = None
        for h in handle_names:
            if drawer in h or h.replace("handle_", "") in drawer:
                handle = h
                break
        if handle is None and handle_names:
            handle = handle_names[0]  # fallback

        # 4. Get locations
        # Assume all objects have a 'location' or use their position as needed
        # For this code, we use the positions dict as a proxy for location
        # (In a real system, you may need to map to PDDL location names)
        dice_locations = [positions[d] for d in selected_dice]
        drawer_location = positions[drawer]
        handle_location = positions[handle] if handle else drawer_location

        # 5. Safety Checks: Drawer capacity (simulate as 2 for this example)
        drawer_capacity = 2
        if len(selected_dice) > drawer_capacity:
            print("[Error] Too many dice for the drawer's capacity. Aborting.")
            return

        # 6. Check if drawer is open (simulate by querying environment state)
        # We'll use a try/except in case the state is not available
        try:
            drawer_state = task.get_drawer_state(drawer)
            is_drawer_open = drawer_state.get("open", False)
        except Exception:
            # Fallback: Assume closed
            is_drawer_open = False

        # 7. If drawer is not open, open it using the available skills
        if not is_drawer_open:
            # a) Move to handle location
            try:
                obs, reward, done = execute_go(
                    env, task, from_location=None, to_location=handle_location
                )
            except Exception as e:
                print(f"[Error] Failed to move to handle: {e}")
                return

            # b) Pick the handle
            try:
                obs, reward, done = execute_pick(
                    env, task, object_name=handle, location=handle_location
                )
            except Exception as e:
                print(f"[Error] Failed to pick handle: {e}")
                return

            # c) Pull to open the drawer
            try:
                obs, reward, done = execute_pull(
                    env, task, drawer_name=drawer, handle_name=handle, location=handle_location
                )
            except Exception as e:
                print(f"[Error] Failed to pull and open drawer: {e}")
                return

        # 8. Place each dice in the drawer
        for dice, dice_loc in zip(selected_dice, dice_locations):
            # a) Move to dice location
            try:
                obs, reward, done = execute_go(
                    env, task, from_location=None, to_location=dice_loc
                )
            except Exception as e:
                print(f"[Error] Failed to move to dice {dice}: {e}")
                continue

            # b) Pick the dice
            try:
                obs, reward, done = execute_pick(
                    env, task, object_name=dice, location=dice_loc
                )
            except Exception as e:
                print(f"[Error] Failed to pick dice {dice}: {e}")
                continue

            # c) Move to drawer location
            try:
                obs, reward, done = execute_go(
                    env, task, from_location=None, to_location=drawer_location
                )
            except Exception as e:
                print(f"[Error] Failed to move to drawer for placing dice {dice}: {e}")
                continue

            # d) Place the dice in the drawer
            try:
                obs, reward, done = execute_place(
                    env, task, object_name=dice, drawer_name=drawer, location=drawer_location
                )
            except Exception as e:
                print(f"[Error] Failed to place dice {dice} in drawer: {e}")
                continue

        # 9. (Optional) Push to close the drawer
        try:
            obs, reward, done = execute_push(
                env, task, drawer_name=drawer, location=drawer_location
            )
        except Exception as e:
            print(f"[Warning] Could not push/close the drawer: {e}")

        # === Exploration Phase: Find Missing Predicate ===
        # Try to use exploration skills to discover missing predicates
        # For example, try to identify objects, check temperature, weight, durability, lock
        # Use only available skills: execute_pick, execute_place, execute_push, execute_pull, execute_sweep, execute_rotate, execute_go, execute_gripper

        # We'll attempt to use execute_pull to see if lock-known is missing
        for dice, dice_loc in zip(selected_dice, dice_locations):
            try:
                # Move to dice location
                obs, reward, done = execute_go(
                    env, task, from_location=None, to_location=dice_loc
                )
                # Pick the dice
                obs, reward, done = execute_pick(
                    env, task, object_name=dice, location=dice_loc
                )
                # Try to pull (simulate exploration for lock-known)
                obs, reward, done = execute_pull(
                    env, task, drawer_name=drawer, handle_name=handle, location=drawer_location
                )
                print(f"[Exploration] Attempted to discover lock-known for {drawer} via pull.")
            except Exception as e:
                print(f"[Exploration] Could not perform exploration pull for {drawer}: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
