# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We use the available skills to explore the environment and infer missing predicates.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.

        # We will attempt to use the available skills to interact with objects and drawers,
        # and observe if any action fails or if the environment provides feedback about missing knowledge.

        # Get all objects, drawers, and locations from positions
        objects = []
        drawers = []
        locations = []
        for name, pos in positions.items():
            if 'drawer' in name:
                drawers.append(name)
            elif 'loc' in name or 'location' in name or 'room' in name:
                locations.append(name)
            else:
                objects.append(name)

        # If locations are not explicitly given, infer from object positions
        if not locations:
            # Assume all unique (x, y) positions are locations
            loc_set = set()
            for pos in positions.values():
                loc_set.add(tuple(pos[:2]))
            locations = ['loc_{}'.format(i) for i in range(len(loc_set))]

        # Try to move to each location and interact with objects/drawers
        # Use only available skills: ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # 1. Try to move to each location
        print("[Exploration] Moving to each location to check for missing predicates...")
        current_location = None
        if hasattr(task, 'get_robot_location'):
            current_location = task.get_robot_location()
        elif locations:
            current_location = locations[0]
        else:
            current_location = None

        for loc in locations:
            if current_location != loc:
                try:
                    print(f"[Exploration] execute_go: {current_location} -> {loc}")
                    obs, reward, done = execute_go(env, task, current_location, loc)
                    current_location = loc
                except Exception as e:
                    print(f"[Exploration] execute_go failed: {e}")

        # 2. Try to pick each object on the floor at the current location
        print("[Exploration] Trying to pick objects on the floor...")
        for obj in objects:
            try:
                # Check if object is on the floor at current location
                # This requires environment state, so we use positions as a proxy
                obj_pos = positions.get(obj, None)
                if obj_pos is not None:
                    # Try to pick
                    print(f"[Exploration] execute_pick: {obj} at {current_location}")
                    obs, reward, done = execute_pick(env, task, obj, current_location)
            except Exception as e:
                print(f"[Exploration] execute_pick failed for {obj}: {e}")

        # 3. Try to pull each drawer handle (if handle objects are present)
        print("[Exploration] Trying to pull drawer handles...")
        for drawer in drawers:
            # Find handle object for this drawer
            handle_obj = None
            for obj in objects:
                if 'handle' in obj and drawer in obj:
                    handle_obj = obj
                    break
            if handle_obj:
                try:
                    # Try to pick the handle first
                    print(f"[Exploration] execute_pick (handle): {handle_obj} at {current_location}")
                    obs, reward, done = execute_pick(env, task, handle_obj, current_location)
                except Exception as e:
                    print(f"[Exploration] execute_pick failed for handle {handle_obj}: {e}")
                try:
                    print(f"[Exploration] execute_pull: {drawer}, handle {handle_obj} at {current_location}")
                    obs, reward, done = execute_pull(env, task, drawer, handle_obj, current_location)
                except Exception as e:
                    print(f"[Exploration] execute_pull failed for {drawer}: {e}")

        # 4. Try to open and close drawers
        print("[Exploration] Trying to open and close drawers...")
        for drawer in drawers:
            try:
                print(f"[Exploration] execute_push: {drawer} at {current_location}")
                obs, reward, done = execute_push(env, task, drawer, current_location)
            except Exception as e:
                print(f"[Exploration] execute_push failed for {drawer}: {e}")

        # 5. Try to place objects in drawers (if possible)
        print("[Exploration] Trying to place objects in drawers...")
        for obj in objects:
            for drawer in drawers:
                try:
                    print(f"[Exploration] execute_place: {obj} in {drawer} at {current_location}")
                    obs, reward, done = execute_place(env, task, obj, drawer, current_location)
                except Exception as e:
                    print(f"[Exploration] execute_place failed for {obj} in {drawer}: {e}")

        # 6. Try to sweep objects (if available)
        print("[Exploration] Trying to sweep objects...")
        for obj in objects:
            try:
                print(f"[Exploration] execute_sweep: {obj} at {current_location}")
                obs, reward, done = execute_sweep(env, task, obj, current_location)
            except Exception as e:
                print(f"[Exploration] execute_sweep failed for {obj}: {e}")

        # 7. Try to use gripper (if available)
        print("[Exploration] Trying to use gripper...")
        try:
            obs, reward, done = execute_gripper(env, task)
        except Exception as e:
            print(f"[Exploration] execute_gripper failed: {e}")

        print("[Exploration] Exploration phase complete. Check logs for missing predicate clues.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()