# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback indicates (drawer-closed drawer1) is missing or needs to be checked.
        # We will use available skills to explore the state of drawer1.

        # Assumptions for object names and locations
        # You may need to adjust these names based on your environment
        drawer_name = 'drawer1'
        handle_name = 'handle1'
        robot_location = 'loc_robot'
        drawer_location = 'loc_drawer'
        handle_location = 'loc_handle'

        # Try to get actual positions if available
        try:
            drawer_pos = positions.get(drawer_name, None)
            handle_pos = positions.get(handle_name, None)
            robot_pos = positions.get('robot', None)
        except Exception as e:
            print("[Exploration] Could not retrieve positions:", e)
            drawer_pos = None
            handle_pos = None
            robot_pos = None

        # Step 1: Move robot to drawer location (if not already there)
        try:
            # If robot_pos and drawer_pos are available, use them
            if robot_pos is not None and drawer_pos is not None:
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=robot_pos,
                    to_location=drawer_pos,
                    max_steps=100
                )
            else:
                # Use placeholder location names if positions are not available
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=robot_location,
                    to_location=drawer_location,
                    max_steps=100
                )
            print("[Exploration] Robot moved to drawer location.")
        except Exception as e:
            print("[Exploration] Error during execute_go:", e)

        # Step 2: Try to pull the drawer to check if it is closed/locked/unlocked
        # This will help us determine the missing predicate (drawer-closed)
        try:
            # First, pick the handle if required
            obs, reward, done = execute_pick(
                env,
                task,
                object_name=handle_name,
                location=drawer_location,
                max_steps=100
            )
            print("[Exploration] Picked handle:", handle_name)
        except Exception as e:
            print("[Exploration] Error during execute_pick (handle):", e)

        # Try to pull the drawer using the handle
        try:
            obs, reward, done = execute_pull(
                env,
                task,
                drawer_name=drawer_name,
                handle_name=handle_name,
                location=drawer_location,
                max_steps=100
            )
            print("[Exploration] Attempted to pull the drawer.")
        except Exception as e:
            print("[Exploration] Error during execute_pull:", e)
            print("[Exploration] This may indicate the drawer is closed or locked (missing predicate: drawer-closed).")

        # Step 3: If pull failed, try to push (in case it's open), or otherwise confirm state
        try:
            obs, reward, done = execute_push(
                env,
                task,
                drawer_name=drawer_name,
                location=drawer_location,
                max_steps=100
            )
            print("[Exploration] Attempted to push the drawer closed.")
        except Exception as e:
            print("[Exploration] Error during execute_push:", e)

        # Step 4: Exploration result
        print("[Exploration] If pulling the drawer failed, the predicate (drawer-closed drawer1) is likely missing or not satisfied.")
        print("[Exploration] If pushing succeeded, the drawer was open and is now closed.")

        # === End of Exploration Phase ===

        # === (Optional) Continue with Oracle Plan Execution ===
        # At this point, you would continue with the main plan using the available skills,
        # now that you have explored and possibly resolved the missing predicate.

        # Example: (Commented out, as actual plan steps depend on your task)
        # obs, reward, done = execute_pick(env, task, object_name='object1', location='loc1', max_steps=100)
        # obs, reward, done = execute_place(env, task, object_name='object1', drawer_name='drawer1', location='loc1', max_steps=100)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
