# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is causing planning to fail.
        # We will use available skills to explore the environment and infer which predicate is missing.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.

        # We will attempt to use the available skills to interact with objects and drawers,
        # and observe if any expected state change or feedback is missing, indicating a missing predicate.

        # Get all objects and locations from positions
        object_names = [name for name in positions if positions[name].get('type', '') == 'object']
        drawer_names = [name for name in positions if positions[name].get('type', '') == 'drawer']
        location_names = [name for name in positions if positions[name].get('type', '') == 'location']

        # Fallback: If types are not provided, infer from naming
        if not object_names:
            object_names = [name for name in positions if 'object' in name or 'obj' in name or 'ball' in name or 'handle' in name]
        if not drawer_names:
            drawer_names = [name for name in positions if 'drawer' in name]
        if not location_names:
            location_names = [name for name in positions if 'loc' in name or 'room' in name or 'table' in name or 'floor' in name]

        # Try to find the robot's initial location
        robot_location = None
        for loc in location_names:
            if positions[loc].get('robot_at', False):
                robot_location = loc
                break
        if robot_location is None and location_names:
            robot_location = location_names[0]

        # === Exploration: Try to interact with drawers and objects ===
        # The goal is to see if we can open a drawer, and if not, which predicate is missing (e.g., lock-known, drawer-unlocked, etc.)

        # For each drawer, try to open it using execute_pull
        for drawer in drawer_names:
            # Find handle for this drawer
            handle = None
            for obj in object_names:
                if positions[obj].get('handle_of', None) == drawer:
                    handle = obj
                    break
            if handle is None:
                # Try to find a handle by naming
                for obj in object_names:
                    if 'handle' in obj and drawer in obj:
                        handle = obj
                        break
            if handle is None:
                continue  # Can't find handle, skip

            # Find location of handle/drawer
            drawer_loc = positions[drawer].get('location', None)
            if drawer_loc is None:
                drawer_loc = robot_location  # fallback

            # Move to drawer location if not already there
            if robot_location != drawer_loc:
                try:
                    obs, reward, done = execute_go(env, task, robot_location, drawer_loc)
                    robot_location = drawer_loc
                except Exception as e:
                    print(f"[Exploration] Failed to move to {drawer_loc}: {e}")

            # Try to pick the handle
            try:
                obs, reward, done = execute_pick(env, task, handle, drawer_loc)
            except Exception as e:
                print(f"[Exploration] Failed to pick handle {handle}: {e}")

            # Try to pull (open) the drawer
            try:
                obs, reward, done = execute_pull(env, task, drawer, handle, drawer_loc)
                print(f"[Exploration] Successfully pulled (opened) drawer {drawer} with handle {handle} at {drawer_loc}")
            except Exception as e:
                print(f"[Exploration] Failed to pull drawer {drawer} with handle {handle}: {e}")
                print("[Exploration] This may indicate a missing predicate such as 'drawer-unlocked', 'lock-known', or similar.")

        # Try to pick and place objects into drawers
        for obj in object_names:
            # Find location of object
            obj_loc = positions[obj].get('location', None)
            if obj_loc is None:
                obj_loc = robot_location  # fallback

            # Move to object location if not already there
            if robot_location != obj_loc:
                try:
                    obs, reward, done = execute_go(env, task, robot_location, obj_loc)
                    robot_location = obj_loc
                except Exception as e:
                    print(f"[Exploration] Failed to move to {obj_loc}: {e}")

            # Try to pick the object
            try:
                obs, reward, done = execute_pick(env, task, obj, obj_loc)
                print(f"[Exploration] Successfully picked object {obj} at {obj_loc}")
            except Exception as e:
                print(f"[Exploration] Failed to pick object {obj}: {e}")

            # Try to place the object in each drawer
            for drawer in drawer_names:
                drawer_loc = positions[drawer].get('location', None)
                if drawer_loc is None:
                    drawer_loc = robot_location  # fallback

                # Move to drawer location if not already there
                if robot_location != drawer_loc:
                    try:
                        obs, reward, done = execute_go(env, task, robot_location, drawer_loc)
                        robot_location = drawer_loc
                    except Exception as e:
                        print(f"[Exploration] Failed to move to {drawer_loc}: {e}")

                try:
                    obs, reward, done = execute_place(env, task, obj, drawer, drawer_loc)
                    print(f"[Exploration] Successfully placed object {obj} in drawer {drawer} at {drawer_loc}")
                except Exception as e:
                    print(f"[Exploration] Failed to place object {obj} in drawer {drawer}: {e}")
                    print("[Exploration] This may indicate a missing predicate such as 'drawer-open', 'drawer-unlocked', or similar.")

        # Try to push (close) all drawers
        for drawer in drawer_names:
            drawer_loc = positions[drawer].get('location', None)
            if drawer_loc is None:
                drawer_loc = robot_location  # fallback

            # Move to drawer location if not already there
            if robot_location != drawer_loc:
                try:
                    obs, reward, done = execute_go(env, task, robot_location, drawer_loc)
                    robot_location = drawer_loc
                except Exception as e:
                    print(f"[Exploration] Failed to move to {drawer_loc}: {e}")

            try:
                obs, reward, done = execute_push(env, task, drawer, drawer_loc)
                print(f"[Exploration] Successfully pushed (closed) drawer {drawer} at {drawer_loc}")
            except Exception as e:
                print(f"[Exploration] Failed to push (close) drawer {drawer}: {e}")

        # Try to sweep all objects
        for obj in object_names:
            obj_loc = positions[obj].get('location', None)
            if obj_loc is None:
                obj_loc = robot_location  # fallback

            # Move to object location if not already there
            if robot_location != obj_loc:
                try:
                    obs, reward, done = execute_go(env, task, robot_location, obj_loc)
                    robot_location = obj_loc
                except Exception as e:
                    print(f"[Exploration] Failed to move to {obj_loc}: {e}")

            try:
                obs, reward, done = execute_sweep(env, task, obj, obj_loc)
                print(f"[Exploration] Successfully swept object {obj} at {obj_loc}")
            except Exception as e:
                print(f"[Exploration] Failed to sweep object {obj}: {e}")

        # Try to use gripper skill if available
        try:
            obs, reward, done = execute_gripper(env, task)
            print("[Exploration] Successfully executed gripper skill")
        except Exception as e:
            print(f"[Exploration] Failed to execute gripper skill: {e}")

        print("[Exploration] Exploration phase complete. Review above errors for clues about missing predicates.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
