# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is causing planning to fail.
        # We will use available skills to explore the environment and try to discover missing predicates.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.
        # We will attempt to use the available skills to interact with objects and drawers to trigger any missing state.

        # Get all objects, drawers, and locations from positions
        # Assume positions is a dict: {name: (type, position, ...)}
        objects = []
        drawers = []
        locations = []
        for name, info in positions.items():
            if 'drawer' in name:
                drawers.append(name)
            elif 'loc' in name or 'location' in name:
                locations.append(name)
            else:
                objects.append(name)

        # For robustness, if locations are not explicitly listed, infer from object positions
        if not locations:
            for name, info in positions.items():
                if isinstance(info, dict) and 'location' in info:
                    locations.append(info['location'])
            if not locations:
                # Fallback: use unique positions as locations
                locations = list(set([tuple(info) for info in positions.values()]))

        # Assume robot starts at some location; pick the first location as start
        if locations:
            robot_location = locations[0]
        else:
            robot_location = None

        # Try to move to each location and interact with objects/drawers
        for loc in locations:
            if robot_location != loc:
                try:
                    # Use execute_go to move robot
                    print(f"[Exploration] Moving robot from {robot_location} to {loc}")
                    obs, reward, done = execute_go(env, task, robot_location, loc)
                    robot_location = loc
                except Exception as e:
                    print(f"[Exploration] Failed to move to {loc}: {e}")
                    continue

            # Try to pick up each object at this location
            for obj in objects:
                try:
                    # Check if object is at this location
                    obj_info = positions.get(obj, None)
                    if obj_info is not None:
                        # If object has a 'location' field, check it
                        if isinstance(obj_info, dict) and 'location' in obj_info:
                            if obj_info['location'] != loc:
                                continue
                        # Otherwise, skip location check
                    print(f"[Exploration] Attempting to pick up {obj} at {loc}")
                    obs, reward, done = execute_pick(env, task, obj, loc)
                except Exception as e:
                    print(f"[Exploration] Failed to pick {obj} at {loc}: {e}")

            # Try to interact with drawers at this location
            for drawer in drawers:
                try:
                    # Try to pull the drawer using any handle object
                    for handle in objects:
                        # Check if handle is associated with this drawer
                        # Assume positions[handle] has 'drawer' field if it's a handle
                        handle_info = positions.get(handle, None)
                        if handle_info is not None and isinstance(handle_info, dict):
                            if handle_info.get('drawer', None) == drawer:
                                print(f"[Exploration] Attempting to pull {drawer} with handle {handle} at {loc}")
                                # First, pick the handle if not already holding
                                try:
                                    obs, reward, done = execute_pick(env, task, handle, loc)
                                except Exception as e:
                                    print(f"[Exploration] Could not pick handle {handle}: {e}")
                                # Then, try to pull the drawer
                                try:
                                    obs, reward, done = execute_pull(env, task, drawer, handle, loc)
                                except Exception as e:
                                    print(f"[Exploration] Could not pull drawer {drawer} with handle {handle}: {e}")
                except Exception as e:
                    print(f"[Exploration] Failed to interact with drawer {drawer} at {loc}: {e}")

        # Try to place objects into drawers if possible
        for drawer in drawers:
            for obj in objects:
                for loc in locations:
                    try:
                        print(f"[Exploration] Attempting to place {obj} into {drawer} at {loc}")
                        obs, reward, done = execute_place(env, task, obj, drawer, loc)
                    except Exception as e:
                        print(f"[Exploration] Failed to place {obj} into {drawer} at {loc}: {e}")

        # Try to push drawers closed at each location
        for drawer in drawers:
            for loc in locations:
                try:
                    print(f"[Exploration] Attempting to push drawer {drawer} at {loc}")
                    obs, reward, done = execute_push(env, task, drawer, loc)
                except Exception as e:
                    print(f"[Exploration] Failed to push drawer {drawer} at {loc}: {e}")

        # Try to sweep objects at each location
        for obj in objects:
            for loc in locations:
                try:
                    print(f"[Exploration] Attempting to sweep {obj} at {loc}")
                    obs, reward, done = execute_sweep(env, task, obj, loc)
                except Exception as e:
                    print(f"[Exploration] Failed to sweep {obj} at {loc}: {e}")

        # Try to use gripper action if available
        try:
            print(f"[Exploration] Attempting to use gripper")
            obs, reward, done = execute_gripper(env, task)
        except Exception as e:
            print(f"[Exploration] Failed to use gripper: {e}")

        print("[Exploration] Exploration phase complete. Check logs for missing predicate clues.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()